#!/bin/bash
# ============================================================================
# SeqMaster Installer for Raspberry Pi 5
# ============================================================================
# Usage: sudo ./install.sh
# ============================================================================
set -e

INSTALL_DIR="/opt/seqmaster"
SERVICE_USER="${SUDO_USER:-pi}"

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

echo -e "${GREEN}============================================${NC}"
echo -e "${GREEN}  SeqMaster Installer${NC}"
echo -e "${GREEN}============================================${NC}"

# Must run as root
if [ "$EUID" -ne 0 ]; then
    echo -e "${RED}Error: Run with sudo${NC}"
    echo "Usage: sudo ./install.sh"
    exit 1
fi

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

# ── Step 1: System packages ─────────────────────────────────────────────────
echo -e "\n${YELLOW}[1/5] Installing system dependencies...${NC}"
apt-get update -qq
apt-get install -y -qq python3 python3-venv python3-pip nginx

# ── Step 2: Copy files ──────────────────────────────────────────────────────
echo -e "\n${YELLOW}[2/5] Installing files to $INSTALL_DIR...${NC}"

# Stop existing service if running
systemctl stop seqmaster 2>/dev/null || true

mkdir -p "$INSTALL_DIR"
cp -r "$SCRIPT_DIR/runtime" "$INSTALL_DIR/"
cp -r "$SCRIPT_DIR/interface" "$INSTALL_DIR/"
[ -d "$SCRIPT_DIR/examples" ] && cp -r "$SCRIPT_DIR/examples" "$INSTALL_DIR/"

# Preserve existing data if upgrading
if [ -f "$INSTALL_DIR/runtime/data/station_globals.json.bak" ]; then
    echo "  Preserving existing configuration..."
fi

chown -R "$SERVICE_USER:$SERVICE_USER" "$INSTALL_DIR"
echo -e "  ${GREEN}✓${NC} Files installed"

# ── Step 3: Create Python virtual environment ────────────────────────────────
echo -e "\n${YELLOW}[3/5] Setting up Python environment...${NC}"

if [ ! -d "$INSTALL_DIR/runtime/.venv" ]; then
    sudo -u "$SERVICE_USER" python3 -m venv "$INSTALL_DIR/runtime/.venv"
fi

sudo -u "$SERVICE_USER" "$INSTALL_DIR/runtime/.venv/bin/pip" install --upgrade pip -q
sudo -u "$SERVICE_USER" "$INSTALL_DIR/runtime/.venv/bin/pip" install -r "$INSTALL_DIR/runtime/requirements.txt" -q

echo -e "  ${GREEN}✓${NC} Python dependencies installed"

# ── Step 4: Configure services ───────────────────────────────────────────────
echo -e "\n${YELLOW}[4/5] Configuring services...${NC}"

# Systemd service for backend
cat > /etc/systemd/system/seqmaster.service << SVCEOF
[Unit]
Description=SeqMaster Runtime
After=network.target

[Service]
Type=simple
User=$SERVICE_USER
WorkingDirectory=$INSTALL_DIR/runtime
Environment="PATH=$INSTALL_DIR/runtime/.venv/bin:/usr/bin"
ExecStart=$INSTALL_DIR/runtime/.venv/bin/python -m src.main
Restart=always
RestartSec=5
StandardOutput=journal
StandardError=journal

[Install]
WantedBy=multi-user.target
SVCEOF

# Nginx configuration
cat > /etc/nginx/sites-available/seqmaster << 'NGXEOF'
server {
    listen 80 default_server;
    listen [::]:80 default_server;
    server_name _;

    root /opt/seqmaster/interface;
    index index.html;

    gzip on;
    gzip_types text/plain text/css application/json application/javascript text/xml application/xml;

    location / {
        try_files $uri $uri/ /index.html;
    }

    location /api/ {
        proxy_pass http://127.0.0.1:8000;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_read_timeout 300s;
    }

    location /ws {
        proxy_pass http://127.0.0.1:8000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection "upgrade";
        proxy_set_header Host $host;
        proxy_read_timeout 3600s;
    }
}
NGXEOF

rm -f /etc/nginx/sites-enabled/default
ln -sf /etc/nginx/sites-available/seqmaster /etc/nginx/sites-enabled/seqmaster
echo -e "  ${GREEN}✓${NC} Services configured"

# ── Step 5: Start services ──────────────────────────────────────────────────
echo -e "\n${YELLOW}[5/5] Starting services...${NC}"
systemctl daemon-reload
systemctl enable seqmaster nginx
systemctl restart seqmaster
systemctl restart nginx

sleep 2
BACKEND_STATUS=$(systemctl is-active seqmaster || true)
NGINX_STATUS=$(systemctl is-active nginx || true)

IP_ADDR=$(hostname -I | awk '{print $1}')

echo ""
echo -e "${GREEN}============================================${NC}"
echo -e "${GREEN}  Installation Complete!${NC}"
echo -e "${GREEN}============================================${NC}"
echo ""
echo "  Backend:  $BACKEND_STATUS"
echo "  Nginx:    $NGINX_STATUS"
echo ""
echo -e "  ${CYAN}Access SeqMaster at: http://$IP_ADDR${NC}"
echo ""
echo "  Useful commands:"
echo "    View logs:    sudo journalctl -u seqmaster -f"
echo "    Restart:      sudo systemctl restart seqmaster"
echo "    Stop:         sudo systemctl stop seqmaster"
echo ""
echo "  License activation:"
echo "    1. Go to Settings → License in the web UI"
echo "    2. Copy your Hardware ID"
echo "    3. Purchase at: http://91.98.205.173:3000/buy"
echo ""
