"""
SeqMaster Runtime - Auth API Routes
Layer: API

Authentication and user management endpoints.
"""

from fastapi import APIRouter, Depends, HTTPException, status
from pydantic import BaseModel, Field
from typing import Optional, List
from datetime import datetime

from src.database.connection import get_db_session
from src.database.models import User, UserRole, UserPermission
from src.database.repository import UserRepository
from src.security.auth import (
    authenticate_user, create_access_token, TokenResponse,
    hash_password, require_auth, require_role, get_current_user
)
from src.security.setup import (
    is_setup_complete, get_setup_state, create_initial_admin,
    get_reset_file_instructions
)
from src.security.permissions import (
    Permission, get_all_permissions, get_default_permissions,
    get_permissions_by_category
)

auth_router = APIRouter(tags=["Authentication"])


# ============================================
# REQUEST/RESPONSE MODELS
# ============================================

class LoginRequest(BaseModel):
    """Login request."""
    username: str
    password: str


class InitialSetupRequest(BaseModel):
    """Initial admin setup request."""
    username: str = Field(..., min_length=3, max_length=50)
    password: str = Field(..., min_length=6)
    full_name: Optional[str] = None
    email: Optional[str] = None


class CreateUserRequest(BaseModel):
    """Create user request."""
    username: str = Field(..., min_length=3, max_length=50)
    password: str = Field(..., min_length=6)
    role: UserRole = UserRole.OPERATOR
    full_name: Optional[str] = None
    email: Optional[str] = None


class UpdateUserRequest(BaseModel):
    """Update user request."""
    full_name: Optional[str] = None
    email: Optional[str] = None
    role: Optional[UserRole] = None
    active: Optional[bool] = None
    password: Optional[str] = Field(None, min_length=6)


class SetPermissionRequest(BaseModel):
    """Set user permission request."""
    permission: str
    granted: bool


class ChangePasswordRequest(BaseModel):
    """Change password request."""
    current_password: str
    new_password: str = Field(..., min_length=6)


class UserResponse(BaseModel):
    """User response."""
    id: int
    username: str
    role: str
    full_name: Optional[str]
    email: Optional[str]
    active: bool
    created_at: datetime
    last_login: Optional[datetime]
    permissions: List[dict] = []


# ============================================
# SETUP ENDPOINTS
# ============================================

@auth_router.get("/setup/state")
async def get_system_setup_state(db = Depends(get_db_session)):
    """
    Check if initial setup is required.
    Returns setup state and whether admin needs to be created.
    """
    return await get_setup_state(db)


@auth_router.post("/setup/initial")
async def perform_initial_setup(
    request: InitialSetupRequest,
    db = Depends(get_db_session)
):
    """
    Perform initial system setup - create first admin user.
    
    This endpoint only works when no users exist in the database.
    After the first admin is created, this endpoint will fail.
    """
    if await is_setup_complete(db):
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Initial setup already complete. Use login instead."
        )
    
    try:
        admin = await create_initial_admin(
            db=db,
            username=request.username,
            password=request.password,
            full_name=request.full_name,
            email=request.email
        )
        
        # Auto-login after setup
        token, expires_at = create_access_token(admin)
        
        return {
            "status": "success",
            "message": "Initial setup complete. Admin user created.",
            "access_token": token,
            "token_type": "bearer",
            "expires_at": expires_at.isoformat(),
            "user": {
                "id": admin.id,
                "username": admin.username,
                "role": admin.role.value,
                "full_name": admin.full_name
            }
        }
    except ValueError as e:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e)
        )


@auth_router.get("/setup/reset-instructions")
async def get_password_reset_instructions():
    """
    Get instructions for resetting admin password via SD-card.
    
    This is available even when locked out, as it only provides
    instructions - actual reset requires physical access to SD-card.
    """
    return get_reset_file_instructions()


# ============================================
# AUTHENTICATION ENDPOINTS
# ============================================

@auth_router.post("/login", response_model=TokenResponse)
async def login(request: LoginRequest, db = Depends(get_db_session)):
    """
    Authenticate user and return JWT token.
    """
    # Check if setup is required
    if not await is_setup_complete(db):
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Initial setup required. Please create admin user first."
        )
    
    # Authenticate against database
    user = await authenticate_user(request.username, request.password, db)
    
    if not user:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Invalid username or password",
            headers={"WWW-Authenticate": "Bearer"}
        )
    
    # Create token
    token, expires_at = create_access_token(user)
    
    # Get user permissions
    permissions = await get_user_effective_permissions(user, db)
    
    return TokenResponse(
        access_token=token,
        expires_at=expires_at,
        user={
            "id": user.id,
            "username": user.username,
            "role": user.role.value,
            "full_name": user.full_name,
            "permissions": permissions
        }
    )


@auth_router.get("/me")
async def get_current_user_info(
    user: User = Depends(require_auth),
    db = Depends(get_db_session)
):
    """Get current user information and permissions."""
    permissions = await get_user_effective_permissions(user, db)
    
    return {
        "id": user.id,
        "username": user.username,
        "role": user.role.value,
        "full_name": user.full_name,
        "email": user.email,
        "permissions": permissions
    }


@auth_router.post("/change-password")
async def change_password(
    request: ChangePasswordRequest,
    user: User = Depends(require_auth),
    db = Depends(get_db_session)
):
    """Change current user's password."""
    from src.security.auth import verify_password
    
    if not verify_password(request.current_password, user.password_hash):
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Current password is incorrect"
        )
    
    user.password_hash = hash_password(request.new_password)
    await db.commit()
    
    return {"status": "success", "message": "Password changed successfully"}


# ============================================
# USER MANAGEMENT ENDPOINTS
# ============================================

@auth_router.get("/users")
async def list_users(
    user: User = Depends(require_role(UserRole.ADMIN)),
    db = Depends(get_db_session)
):
    """List all users (Admin only)."""
    repo = UserRepository(db)
    users = await repo.get_all()
    
    return [
        {
            "id": u.id,
            "username": u.username,
            "role": u.role.value,
            "full_name": u.full_name,
            "email": u.email,
            "active": u.active,
            "created_at": u.created_at.isoformat() if u.created_at else None,
            "last_login": u.last_login.isoformat() if u.last_login else None
        }
        for u in users
    ]


@auth_router.post("/users")
async def create_user(
    request: CreateUserRequest,
    user: User = Depends(require_role(UserRole.ADMIN)),
    db = Depends(get_db_session)
):
    """Create a new user (Admin only)."""
    repo = UserRepository(db)
    
    # Check if username exists
    existing = await repo.get_by_username(request.username)
    if existing:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Username already exists"
        )
    
    # Create user
    new_user = User(
        username=request.username,
        password_hash=hash_password(request.password),
        role=request.role,
        full_name=request.full_name,
        email=request.email,
        active=True
    )
    
    db.add(new_user)
    await db.commit()
    await db.refresh(new_user)
    
    return {
        "id": new_user.id,
        "username": new_user.username,
        "role": new_user.role.value,
        "full_name": new_user.full_name,
        "message": "User created successfully"
    }


@auth_router.get("/users/{user_id}")
async def get_user(
    user_id: int,
    current_user: User = Depends(require_role(UserRole.ADMIN)),
    db = Depends(get_db_session)
):
    """Get user details (Admin only)."""
    repo = UserRepository(db)
    user = await repo.get_by_id(user_id)
    
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    
    # Get custom permissions
    permissions = await get_user_custom_permissions(user, db)
    
    return {
        "id": user.id,
        "username": user.username,
        "role": user.role.value,
        "full_name": user.full_name,
        "email": user.email,
        "active": user.active,
        "created_at": user.created_at.isoformat() if user.created_at else None,
        "last_login": user.last_login.isoformat() if user.last_login else None,
        "custom_permissions": permissions
    }


@auth_router.put("/users/{user_id}")
async def update_user(
    user_id: int,
    request: UpdateUserRequest,
    current_user: User = Depends(require_role(UserRole.ADMIN)),
    db = Depends(get_db_session)
):
    """Update user (Admin only)."""
    repo = UserRepository(db)
    user = await repo.get_by_id(user_id)
    
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    
    # Prevent self-demotion for last admin
    if user.id == current_user.id and request.role and request.role != UserRole.ADMIN:
        # Check if there are other admins
        from sqlalchemy import select, func
        result = await db.execute(
            select(func.count(User.id)).where(
                User.role == UserRole.ADMIN,
                User.active == True,
                User.id != user.id
            )
        )
        admin_count = result.scalar()
        if admin_count == 0:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="Cannot demote the last active admin"
            )
    
    # Update fields
    if request.full_name is not None:
        user.full_name = request.full_name
    if request.email is not None:
        user.email = request.email
    if request.role is not None:
        user.role = request.role
    if request.active is not None:
        user.active = request.active
    if request.password:
        user.password_hash = hash_password(request.password)
    
    await db.commit()
    
    return {"status": "success", "message": "User updated successfully"}


@auth_router.delete("/users/{user_id}")
async def delete_user(
    user_id: int,
    current_user: User = Depends(require_role(UserRole.ADMIN)),
    db = Depends(get_db_session)
):
    """Delete user (Admin only)."""
    if user_id == current_user.id:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Cannot delete yourself"
        )
    
    repo = UserRepository(db)
    user = await repo.get_by_id(user_id)
    
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    
    await db.delete(user)
    await db.commit()
    
    return {"status": "success", "message": "User deleted successfully"}


# ============================================
# PERMISSION MANAGEMENT ENDPOINTS
# ============================================

@auth_router.get("/permissions")
async def list_all_permissions(user: User = Depends(require_auth)):
    """List all available permissions with metadata."""
    return {
        "permissions": get_all_permissions(),
        "by_category": get_permissions_by_category()
    }


@auth_router.get("/users/{user_id}/permissions")
async def get_user_permissions(
    user_id: int,
    current_user: User = Depends(require_role(UserRole.ADMIN)),
    db = Depends(get_db_session)
):
    """Get user's effective permissions (Admin only)."""
    repo = UserRepository(db)
    user = await repo.get_by_id(user_id)
    
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    
    # Get default permissions for role
    default_perms = get_default_permissions(user.role)
    
    # Get custom permissions
    custom_perms = await get_user_custom_permissions(user, db)
    
    # Calculate effective permissions
    effective = await get_user_effective_permissions(user, db)
    
    return {
        "user_id": user_id,
        "role": user.role.value,
        "default_permissions": [p.value for p in default_perms],
        "custom_permissions": custom_perms,
        "effective_permissions": effective
    }


@auth_router.post("/users/{user_id}/permissions")
async def set_user_permission(
    user_id: int,
    request: SetPermissionRequest,
    current_user: User = Depends(require_role(UserRole.ADMIN)),
    db = Depends(get_db_session)
):
    """Set a custom permission for a user (Admin only)."""
    repo = UserRepository(db)
    user = await repo.get_by_id(user_id)
    
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    
    # Validate permission exists
    try:
        perm = Permission(request.permission)
    except ValueError:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"Invalid permission: {request.permission}"
        )
    
    # Find existing custom permission
    from sqlalchemy import select
    result = await db.execute(
        select(UserPermission).where(
            UserPermission.user_id == user.id,
            UserPermission.permission == request.permission
        )
    )
    existing = result.scalar_one_or_none()
    
    if existing:
        existing.granted = request.granted
        existing.granted_by = current_user.username
        existing.granted_at = datetime.utcnow()
    else:
        new_perm = UserPermission(
            user_id=user.id,
            permission=request.permission,
            granted=request.granted,
            granted_by=current_user.username
        )
        db.add(new_perm)
    
    await db.commit()
    
    return {
        "status": "success",
        "message": f"Permission '{request.permission}' {'granted' if request.granted else 'denied'}"
    }


@auth_router.delete("/users/{user_id}/permissions/{permission}")
async def remove_user_permission(
    user_id: int,
    permission: str,
    current_user: User = Depends(require_role(UserRole.ADMIN)),
    db = Depends(get_db_session)
):
    """Remove custom permission (revert to role default)."""
    from sqlalchemy import select
    
    result = await db.execute(
        select(UserPermission).where(
            UserPermission.user_id == user_id,
            UserPermission.permission == permission
        )
    )
    existing = result.scalar_one_or_none()
    
    if existing:
        await db.delete(existing)
        await db.commit()
    
    return {"status": "success", "message": "Custom permission removed, using role default"}


# ============================================
# HELPER FUNCTIONS
# ============================================

async def get_user_custom_permissions(user: User, db) -> List[dict]:
    """Get user's custom permission overrides."""
    from sqlalchemy import select
    
    result = await db.execute(
        select(UserPermission).where(UserPermission.user_id == user.id)
    )
    perms = result.scalars().all()
    
    return [
        {
            "permission": p.permission,
            "granted": p.granted,
            "granted_by": p.granted_by,
            "granted_at": p.granted_at.isoformat() if p.granted_at else None
        }
        for p in perms
    ]


async def get_user_effective_permissions(user: User, db) -> List[str]:
    """Calculate user's effective permissions (role defaults + custom)."""
    # Start with role defaults
    default_perms = get_default_permissions(user.role)
    effective = set(p.value for p in default_perms)
    
    # Apply custom overrides
    custom = await get_user_custom_permissions(user, db)
    for cp in custom:
        if cp["granted"]:
            effective.add(cp["permission"])
        else:
            effective.discard(cp["permission"])
    
    return sorted(list(effective))
