"""
SeqMaster Runtime - License API
Layer: API

REST API endpoints for license management.

Endpoints:
- GET  /license/status     - Get current license status
- GET  /license/hardware   - Get hardware ID for activation
- POST /license/upload     - Upload license file
- DELETE /license          - Remove license (return to demo)
- GET  /license/features   - List all features and their status
"""

from fastapi import APIRouter, HTTPException, UploadFile, File, Depends
from pydantic import BaseModel
from typing import Dict, List, Optional
import logging

from src.core.license import (
    get_license_service, 
    LicenseInfo, 
    LicenseTier,
    FEATURE_DESCRIPTIONS,
    DEMO_LIMITS,
)
from src.core.fingerprint import get_hardware_fingerprint
from src.core.config import settings
from src.security.auth import get_current_user, require_permission

logger = logging.getLogger(__name__)

license_router = APIRouter(prefix="/license", tags=["license"])


# ============================================
# RESPONSE MODELS
# ============================================

class LicenseStatusResponse(BaseModel):
    """License status response."""
    tier: str
    tier_display: str
    is_valid: bool
    is_demo: bool
    license_id: Optional[str] = None
    customer_name: Optional[str] = None
    customer_email: Optional[str] = None
    hardware_id: str
    features: List[str]
    issued_at: Optional[str] = None
    expires_at: Optional[str] = None
    days_remaining: Optional[int] = None
    error: Optional[str] = None
    demo_limits: Optional[Dict[str, int]] = None


class HardwareInfoResponse(BaseModel):
    """Hardware information for license activation."""
    hardware_id: str
    cpu_serial: Optional[str]
    mac_address: Optional[str]
    purchase_url: str


class FeatureInfo(BaseModel):
    """Feature information."""
    name: str
    description: str
    enabled: bool


class FeaturesResponse(BaseModel):
    """All features response."""
    features: List[FeatureInfo]


class UploadResponse(BaseModel):
    """License upload response."""
    success: bool
    message: str
    tier: Optional[str] = None


# ============================================
# HELPER FUNCTIONS
# ============================================

TIER_DISPLAY_NAMES = {
    LicenseTier.DEMO: "Demo",
    LicenseTier.BASIC: "Basic",
    LicenseTier.PROFESSIONAL: "Professional",
    LicenseTier.ENTERPRISE: "Enterprise",
}


def get_purchase_url(hardware_id: str) -> str:
    """Generate purchase URL with hardware ID."""
    # Konfigurer din egen URL her
    base_url = getattr(settings, 'LICENSE_PURCHASE_URL', 'http://91.98.205.173:3000/buy')
    return f"{base_url}?hardware_id={hardware_id}"


# ============================================
# ENDPOINTS
# ============================================

@license_router.get("/status", response_model=LicenseStatusResponse)
async def get_license_status():
    """
    Get current license status.
    
    Returns license tier, validity, features, and expiration info.
    Available to all users (no authentication required for basic info).
    """
    service = get_license_service()
    info = service.get_license_info()
    
    return LicenseStatusResponse(
        tier=info.tier.value,
        tier_display=TIER_DISPLAY_NAMES.get(info.tier, info.tier.value),
        is_valid=info.is_valid,
        is_demo=service.is_demo(),
        license_id=info.license_id,
        customer_name=info.customer_name,
        customer_email=info.customer_email,
        hardware_id=info.hardware_id or get_hardware_fingerprint()["hardware_id"],
        features=info.features,
        issued_at=info.issued_at.isoformat() if info.issued_at else None,
        expires_at=info.expires_at.isoformat() if info.expires_at else None,
        days_remaining=info._days_remaining() if info.expires_at else None,
        error=info.error,
        demo_limits=DEMO_LIMITS if service.is_demo() else None,
    )


@license_router.get("/hardware", response_model=HardwareInfoResponse)
async def get_hardware_info():
    """
    Get hardware information for license activation.
    
    Returns the hardware ID needed to purchase a license,
    along with a direct link to the purchase page.
    """
    fingerprint = get_hardware_fingerprint()
    hardware_id = fingerprint["hardware_id"]
    
    return HardwareInfoResponse(
        hardware_id=hardware_id,
        cpu_serial=fingerprint.get("cpu_serial"),
        mac_address=fingerprint.get("mac_address"),
        purchase_url=get_purchase_url(hardware_id),
    )


@license_router.get("/features", response_model=FeaturesResponse)
async def get_features():
    """
    Get all available features and their current status.
    
    Returns a list of all features with descriptions and
    whether they are enabled for the current license.
    """
    service = get_license_service()
    
    features = []
    for name, description in FEATURE_DESCRIPTIONS.items():
        features.append(FeatureInfo(
            name=name,
            description=description,
            enabled=service.has_feature(name),
        ))
    
    return FeaturesResponse(features=features)


@license_router.post("/upload", response_model=UploadResponse)
async def upload_license(
    file: UploadFile = File(...),
    current_user = Depends(require_permission("system_config"))
):
    """
    Upload and activate a license file.
    
    Requires system_config permission (Admin only).
    
    The file should be the .sig file received after purchase.
    It will be validated against:
    - Digital signature (must be signed by SeqMaster)
    - Hardware ID (must match this device)
    - Expiration date (must not be expired)
    """
    # Valider filtype
    if not file.filename:
        raise HTTPException(
            status_code=400, 
            detail="Ingen fil modtaget"
        )
    
    if not file.filename.endswith('.sig'):
        logger.warning(f"Ugyldig filtype: {file.filename}")
        # Accepter alligevel, men log det
    
    try:
        # Læs fil indhold
        content = await file.read()
        
        if len(content) > 1024 * 100:  # Max 100KB
            raise HTTPException(
                status_code=400,
                detail="Licensfil er for stor (max 100KB)"
            )
        
        # Upload og valider
        service = get_license_service()
        info = service.upload_license(content)
        
        logger.info(f"Licens uploadet af {current_user.username}: {info.license_id}")
        
        return UploadResponse(
            success=True,
            message=f"Licens aktiveret! Niveau: {TIER_DISPLAY_NAMES.get(info.tier, info.tier.value)}",
            tier=info.tier.value,
        )
        
    except ValueError as e:
        logger.warning(f"Licens upload fejlede: {e}")
        raise HTTPException(
            status_code=400,
            detail=str(e)
        )
    except Exception as e:
        logger.exception("Uventet fejl ved licens upload")
        raise HTTPException(
            status_code=500,
            detail="Intern fejl ved behandling af licensfil"
        )


@license_router.delete("", response_model=UploadResponse)
async def remove_license(
    current_user = Depends(require_permission("system_config"))
):
    """
    Remove current license and return to demo mode.
    
    Requires system_config permission (Admin only).
    
    This is useful for:
    - Testing demo mode
    - Transferring license to another device
    - Troubleshooting license issues
    """
    service = get_license_service()
    
    if service.remove_license():
        logger.info(f"Licens fjernet af {current_user.username}")
        return UploadResponse(
            success=True,
            message="Licens fjernet. Systemet kører nu i demo mode.",
            tier=LicenseTier.DEMO.value,
        )
    else:
        return UploadResponse(
            success=False,
            message="Ingen licens at fjerne.",
            tier=LicenseTier.DEMO.value,
        )
