"""
Network management API endpoints for WiFi configuration.
"""

from fastapi import APIRouter, HTTPException
from typing import List, Dict, Any
import subprocess
import re
import json

router = APIRouter(prefix="/network", tags=["network"])


@router.get("/wifi/scan")
async def scan_wifi() -> List[Dict[str, Any]]:
    """Scan for available WiFi networks."""
    try:
        # Use nmcli to scan WiFi
        result = subprocess.run(
            ["nmcli", "-t", "-f", "SSID,SIGNAL,SECURITY", "dev", "wifi", "list"],
            capture_output=True,
            text=True,
            timeout=10
        )
        
        # Get saved connections
        saved_result = subprocess.run(
            ["nmcli", "-t", "-f", "NAME,TYPE", "connection", "show"],
            capture_output=True,
            text=True,
            timeout=5
        )
        
        saved_ssids = set()
        for line in saved_result.stdout.strip().split('\n'):
            if not line:
                continue
            parts = line.split(':')
            if len(parts) >= 2 and parts[1] == '802-11-wireless':
                saved_ssids.add(parts[0])
        
        networks = []
        seen_ssids = set()
        
        for line in result.stdout.strip().split('\n'):
            if not line:
                continue
            
            parts = line.split(':')
            if len(parts) >= 3:
                ssid = parts[0]
                signal = parts[1]
                security = parts[2]
                
                # Skip duplicates and empty SSIDs
                if ssid and ssid not in seen_ssids:
                    seen_ssids.add(ssid)
                    networks.append({
                        "ssid": ssid,
                        "signal": int(signal) if signal.isdigit() else 0,
                        "security": security,
                        "secured": bool(security and security != '--'),
                        "saved": ssid in saved_ssids
                    })
        
        # Sort by signal strength
        networks.sort(key=lambda x: x['signal'], reverse=True)
        
        return networks
    except subprocess.TimeoutExpired:
        raise HTTPException(status_code=408, detail="WiFi scan timeout")
    except FileNotFoundError:
        raise HTTPException(status_code=503, detail="NetworkManager not available")
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"WiFi scan failed: {str(e)}")


@router.get("/wifi/status")
async def get_wifi_status() -> Dict[str, Any]:
    """Get current WiFi connection status."""
    try:
        # Get connection status
        result = subprocess.run(
            ["nmcli", "-t", "-f", "TYPE,DEVICE,STATE,NAME", "connection", "show", "--active"],
            capture_output=True,
            text=True,
            timeout=5
        )
        
        wifi_connected = False
        current_ssid = None
        wifi_device = None
        eth_device = None
        
        for line in result.stdout.strip().split('\n'):
            if not line:
                continue
            parts = line.split(':')
            if len(parts) >= 4 and parts[0] == '802-11-wireless' and parts[2] == 'activated':
                wifi_connected = True
                wifi_device = parts[1]
                current_ssid = parts[3]
            elif len(parts) >= 3 and parts[0] == '802-3-ethernet' and parts[2] == 'activated':
                eth_device = parts[1]
        
        ethernet_connected = bool(eth_device)
        
        # Get WiFi signal strength if connected
        signal = None
        if wifi_connected and wifi_device:
            signal_result = subprocess.run(
                ["nmcli", "-t", "-f", "SIGNAL", "dev", "wifi", "list"],
                capture_output=True,
                text=True,
                timeout=5
            )
            lines = signal_result.stdout.strip().split('\n')
            if lines and lines[0]:
                try:
                    signal = int(lines[0])
                except ValueError:
                    pass
        
        # Get detailed network info for both interfaces
        wifi_info = None
        eth_info = None
        
        if wifi_device:
            wifi_info = get_interface_info(wifi_device)
        if eth_device:
            eth_info = get_interface_info(eth_device)
        
        return {
            "wifi_connected": wifi_connected,
            "ethernet_connected": ethernet_connected,
            "current_ssid": current_ssid,
            "device": wifi_device,
            "signal": signal,
            "wifi_info": wifi_info,
            "ethernet_info": eth_info
        }
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Failed to get WiFi status: {str(e)}")


def get_interface_info(interface: str) -> Dict[str, Any]:
    """Get detailed information about a network interface."""
    try:
        # Get IP address and method (dhcp/manual)
        result = subprocess.run(
            ["nmcli", "-t", "-f", "IP4.ADDRESS,IP4.GATEWAY,IP4.DNS,GENERAL.HWADDR", "dev", "show", interface],
            capture_output=True,
            text=True,
            timeout=5
        )
        
        info = {
            "interface": interface,
            "ip_address": None,
            "gateway": None,
            "dns": [],
            "mac_address": None,
            "method": "dhcp"  # Will be determined from connection settings
        }
        
        for line in result.stdout.strip().split('\n'):
            if not line:
                continue
            if line.startswith('IP4.ADDRESS'):
                # Format: IP4.ADDRESS[1]:192.168.1.100/24
                parts = line.split(':', 1)
                if len(parts) > 1:
                    info["ip_address"] = parts[1].strip()
            elif line.startswith('IP4.GATEWAY'):
                parts = line.split(':', 1)
                if len(parts) > 1:
                    info["gateway"] = parts[1].strip()
            elif line.startswith('IP4.DNS'):
                parts = line.split(':', 1)
                if len(parts) > 1:
                    info["dns"].append(parts[1].strip())
            elif line.startswith('GENERAL.HWADDR'):
                parts = line.split(':', 1)
                if len(parts) > 1:
                    info["mac_address"] = parts[1].strip()
        
        # Get connection method (dhcp vs manual)
        conn_result = subprocess.run(
            ["nmcli", "-t", "-f", "name", "con", "show", "--active"],
            capture_output=True,
            text=True,
            timeout=5
        )
        
        for conn_name in conn_result.stdout.strip().split('\n'):
            if not conn_name:
                continue
            method_result = subprocess.run(
                ["nmcli", "-t", "-f", "ipv4.method", "con", "show", conn_name],
                capture_output=True,
                text=True,
                timeout=5
            )
            if method_result.returncode == 0:
                method_line = method_result.stdout.strip()
                if 'manual' in method_line:
                    info["method"] = "static"
                break
        
        return info
    except Exception as e:
        return {
            "interface": interface,
            "error": str(e)
        }


@router.post("/wifi/connect")
async def connect_wifi(request: Dict[str, str]) -> Dict[str, str]:
    """Connect to a WiFi network."""
    ssid = request.get("ssid")
    password = request.get("password", "")
    
    if not ssid:
        raise HTTPException(status_code=400, detail="SSID is required")
    
    try:
        # Delete existing connection if it exists
        subprocess.run(
            ["nmcli", "connection", "delete", ssid],
            capture_output=True,
            text=True
        )
        
        # Get WiFi interface name
        iface_result = subprocess.run(
            ["nmcli", "-t", "-f", "DEVICE,TYPE", "device"],
            capture_output=True,
            text=True
        )
        
        wifi_iface = "wlan0"  # default
        for line in iface_result.stdout.strip().split('\n'):
            parts = line.split(':')
            if len(parts) >= 2 and parts[1] == 'wifi':
                wifi_iface = parts[0]
                break
        
        # Create new connection with explicit settings
        if password:
            # Create connection with WPA-PSK security
            subprocess.run(
                ["nmcli", "con", "add", 
                 "type", "wifi",
                 "con-name", ssid,
                 "ifname", wifi_iface,
                 "ssid", ssid,
                 "wifi-sec.key-mgmt", "wpa-psk",
                 "wifi-sec.psk", password],
                capture_output=True,
                text=True,
                timeout=10,
                check=True
            )
        else:
            # Create open network connection
            subprocess.run(
                ["nmcli", "con", "add",
                 "type", "wifi",
                 "con-name", ssid,
                 "ifname", wifi_iface,
                 "ssid", ssid],
                capture_output=True,
                text=True,
                timeout=10,
                check=True
            )
        
        # Activate the connection
        result = subprocess.run(
            ["nmcli", "con", "up", ssid],
            capture_output=True,
            text=True,
            timeout=30
        )
        
        if result.returncode == 0:
            return {"status": "connected", "ssid": ssid}
        else:
            error_msg = result.stderr or result.stdout
            raise HTTPException(status_code=400, detail=f"Connection failed: {error_msg}")
            
    except subprocess.TimeoutExpired:
        raise HTTPException(status_code=408, detail="Connection timeout")
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Failed to connect: {str(e)}")


@router.delete("/wifi/disconnect")
async def disconnect_wifi() -> Dict[str, str]:
    """Disconnect from current WiFi network."""
    try:
        # Get current WiFi connection
        result = subprocess.run(
            ["nmcli", "-t", "-f", "NAME,TYPE", "connection", "show", "--active"],
            capture_output=True,
            text=True,
            timeout=5
        )
        
        for line in result.stdout.strip().split('\n'):
            if not line:
                continue
            parts = line.split(':')
            if len(parts) >= 2 and parts[1] == '802-11-wireless':
                connection_name = parts[0]
                subprocess.run(
                    ["nmcli", "connection", "down", connection_name],
                    capture_output=True,
                    text=True,
                    timeout=10,
                    check=True
                )
                return {"status": "disconnected"}
        
        return {"status": "not_connected"}
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Failed to disconnect: {str(e)}")


@router.get("/wifi/saved")
async def get_saved_networks() -> List[Dict[str, Any]]:
    """Get list of saved WiFi networks."""
    try:
        result = subprocess.run(
            ["nmcli", "-t", "-f", "NAME,TYPE,AUTOCONNECT", "connection", "show"],
            capture_output=True,
            text=True,
            timeout=5
        )
        
        networks = []
        for line in result.stdout.strip().split('\n'):
            if not line:
                continue
            parts = line.split(':')
            if len(parts) >= 3 and parts[1] == '802-11-wireless':
                networks.append({
                    "ssid": parts[0],
                    "autoconnect": parts[2] == 'yes'
                })
        
        return networks
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Failed to get saved networks: {str(e)}")


@router.delete("/wifi/forget/{ssid}")
async def forget_network(ssid: str) -> Dict[str, str]:
    """Forget (delete) a saved WiFi network."""
    try:
        result = subprocess.run(
            ["nmcli", "connection", "delete", ssid],
            capture_output=True,
            text=True,
            timeout=10
        )
        
        if result.returncode == 0:
            return {"status": "forgotten", "ssid": ssid}
        else:
            raise HTTPException(status_code=404, detail="Network not found")
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Failed to forget network: {str(e)}")


@router.post("/configure-ip")
async def configure_ip(request: Dict[str, Any]) -> Dict[str, str]:
    """Configure IP address settings for an interface."""
    interface = request.get("interface")
    method = request.get("method", "dhcp")  # dhcp or static
    ip_address = request.get("ip_address")
    gateway = request.get("gateway")
    dns = request.get("dns", [])
    
    if not interface:
        raise HTTPException(status_code=400, detail="Interface is required")
    
    try:
        # Get active connection name for this interface
        conn_result = subprocess.run(
            ["nmcli", "-t", "-f", "NAME,DEVICE", "con", "show", "--active"],
            capture_output=True,
            text=True,
            timeout=5
        )
        
        connection_name = None
        for line in conn_result.stdout.strip().split('\n'):
            if not line:
                continue
            parts = line.split(':')
            if len(parts) >= 2 and parts[1] == interface:
                connection_name = parts[0]
                break
        
        if not connection_name:
            raise HTTPException(status_code=404, detail=f"No active connection found for {interface}")
        
        # Configure based on method
        if method == "dhcp":
            subprocess.run(
                ["nmcli", "con", "modify", connection_name, "ipv4.method", "auto"],
                capture_output=True,
                text=True,
                timeout=10,
                check=True
            )
        elif method == "static":
            if not ip_address:
                raise HTTPException(status_code=400, detail="IP address required for static configuration")
            
            # Add /24 subnet mask if not provided
            if '/' not in ip_address:
                ip_address = f"{ip_address}/24"
            
            # Set method to manual and IP address in one command
            subprocess.run(
                ["nmcli", "con", "modify", connection_name, 
                 "ipv4.method", "manual",
                 "ipv4.addresses", ip_address],
                capture_output=True,
                text=True,
                timeout=10,
                check=True
            )
            
            # Set gateway if provided
            if gateway:
                subprocess.run(
                    ["nmcli", "con", "modify", connection_name, "ipv4.gateway", gateway],
                    capture_output=True,
                    text=True,
                    timeout=10,
                    check=True
                )
            
            # Set DNS if provided
            if dns:
                dns_str = ','.join(dns)
                subprocess.run(
                    ["nmcli", "con", "modify", connection_name, "ipv4.dns", dns_str],
                    capture_output=True,
                    text=True,
                    timeout=10,
                    check=True
                )
        
        # Restart connection to apply changes
        subprocess.run(
            ["nmcli", "con", "down", connection_name],
            capture_output=True,
            text=True,
            timeout=10
        )
        
        subprocess.run(
            ["nmcli", "con", "up", connection_name],
            capture_output=True,
            text=True,
            timeout=30,
            check=True
        )
        
        return {"status": "configured", "interface": interface, "method": method}
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Failed to configure IP: {str(e)}")
