"""
SeqMaster Runtime - API Routes
Layer: API

REST API endpoints for the SeqMaster runtime.

FILE OVERVIEW (for AI context):
- Lines 1-80: Imports, request/response models
- Lines 80-170: System endpoints (/system/info, /capabilities, etc.)
- Lines 170-450: Sequence CRUD (/sequences/*)
- Lines 450-650: Execution control (/execution/*)
- Lines 650-900: Test results (/results/*)
- Lines 900+: Property sets, misc endpoints
"""

import json
import yaml
from datetime import datetime
from pathlib import Path
from typing import Any, Dict, List, Optional

from fastapi import APIRouter, Depends, HTTPException, status, UploadFile, File
from sqlalchemy import delete
from pydantic import BaseModel

from src.database.connection import get_db_session
from src.database.repository import (
    SequenceRepository, PropertySetRepository, 
    TestSessionRepository, StepResultRepository
)
from src.core.service_registry import ServiceRegistry
from src.security.auth import get_current_user, require_permission
from src.database.models import UserRole, TestStatus, StepResult
from src.api.serializers import isoformat_z, duration_seconds, session_summary, serialize_step_result
from src.core.license import get_license_service, DEMO_LIMITS

api_router = APIRouter()


# ============================================
# UTILITY FUNCTIONS
# ============================================

def parse_sequence_content(content: str) -> dict:
    """
    Parse sequence content as JSON or YAML.
    
    Args:
        content: Raw sequence content string
        
    Returns:
        Parsed dictionary
        
    Raises:
        ValueError: If content cannot be parsed as JSON or YAML
    """
    try:
        return json.loads(content)
    except json.JSONDecodeError:
        try:
            return yaml.safe_load(content)
        except yaml.YAMLError as e:
            raise ValueError(f"Content is neither valid JSON nor YAML: {e}")


# ============================================
# REQUEST/RESPONSE MODELS
# ============================================

class SystemInfoResponse(BaseModel):
    """System information response."""
    tester_id: str
    hostname: str
    version: str
    uptime_seconds: float
    capabilities: Dict[str, Any]


class SequenceUploadRequest(BaseModel):
    """Sequence upload request."""
    sequence_id: str
    version: str
    name: str
    content: str
    dut_type: Optional[str] = None
    description: Optional[str] = None


class PropertySetUploadRequest(BaseModel):
    """Property set upload request."""
    property_set_id: str
    version: str
    name: str
    content: str
    dut_type: str
    dut_revision: Optional[str] = None


class ExecutionStartRequest(BaseModel):
    """Start execution request."""
    sequence_id: str
    sequence_version: Optional[str] = None
    dut_id: str
    dut_serial: Optional[str] = None
    property_set_id: Optional[str] = None
    dry_run: bool = False
    scanned_fields: Optional[Dict[str, Any]] = None  # Fields extracted from barcode scanner


class ExecutionControlRequest(BaseModel):
    """Execution control request."""
    action: str  # pause, resume, abort


class OperatorInputRequest(BaseModel):
    """Operator input request."""
    session_id: str
    input_type: str
    value: Any


class StorageCategory(BaseModel):
    """Storage category info."""
    name: str
    size_bytes: int
    color: str


class StorageInfoResponse(BaseModel):
    """Storage info response."""
    total_bytes: int
    used_bytes: int
    free_bytes: int
    categories: list[StorageCategory]


# --- Tester Lock Models ---

class TesterAcquireRequest(BaseModel):
    """Request to acquire tester lock."""
    client_id: str
    operator_id: Optional[str] = None
    operator_name: Optional[str] = None
    force: bool = False


class TesterReleaseRequest(BaseModel):
    """Request to release tester lock."""
    client_id: str
    force: bool = False


class TesterHeartbeatRequest(BaseModel):
    """Heartbeat to keep lock alive."""
    client_id: str


# --- Sequence Match Models ---

class SequenceMatchRequest(BaseModel):
    """Request to find matching sequence for DUT."""
    barcode: str
    extracted_variables: Optional[Dict[str, Any]] = None
    include_drafts: bool = False  # True for admin/engineer to include draft sequences


class MatchedSequenceInfo(BaseModel):
    """Info about a matched sequence."""
    sequence_id: str
    sequence_name: str
    version: str
    match_type: str
    confidence: int


class SequenceMatchResponse(BaseModel):
    """Response with matched sequence info."""
    matched: bool
    sequence_id: Optional[str] = None
    sequence_name: Optional[str] = None
    version: Optional[str] = None
    match_type: Optional[str] = None  # dut_filter, exact_partnumber, pattern, hw_version, none
    confidence: int = 0
    message: Optional[str] = None
    auto_start: bool = False  # True if exactly one sequence matches dut_filters
    matching_sequences: List[MatchedSequenceInfo] = []  # All matching sequences


# ============================================
# SYSTEM ENDPOINTS
# ============================================

@api_router.get("/system/info", response_model=SystemInfoResponse)
async def get_system_info():
    """Get system information and capabilities."""
    from src.core.config import settings
    import time
    
    capabilities = ServiceRegistry.get("capabilities")
    start_time = ServiceRegistry.get("start_time") or time.time()
    
    return SystemInfoResponse(
        tester_id=settings.TESTER_ID,
        hostname=settings.HOSTNAME,
        version=settings.VERSION,
        uptime_seconds=time.time() - start_time,
        capabilities=capabilities.to_dict() if capabilities else {}
    )


@api_router.get("/system/capabilities")
async def get_capabilities():
    """Get hardware capabilities."""
    capabilities = ServiceRegistry.get("capabilities")
    return capabilities.to_dict() if capabilities else {}


@api_router.post("/system/refresh-hardware")
async def refresh_hardware(user = Depends(require_permission("configure_hardware"))):
    """Refresh hardware detection. Requires: configure_hardware"""
    detector = ServiceRegistry.get("hardware_detector")
    if detector:
        capabilities = await detector.refresh()
        ServiceRegistry.set("capabilities", capabilities)
        return capabilities.to_dict()
    raise HTTPException(status_code=500, detail="Hardware detector not available")


@api_router.post("/system/shutdown")
async def shutdown_system(user = Depends(require_permission("shutdown_reboot"))):
    """Shutdown the system. Requires: shutdown_reboot"""
    import subprocess
    import asyncio
    
    # Schedule shutdown in 5 seconds to allow response to be sent
    async def do_shutdown():
        await asyncio.sleep(2)
        subprocess.run(["sudo", "shutdown", "-h", "now"], check=False)
    
    asyncio.create_task(do_shutdown())
    return {"status": "shutting_down", "message": "System will shutdown in 2 seconds"}


@api_router.post("/system/reboot")
async def reboot_system(user = Depends(require_permission("shutdown_reboot"))):
    """Reboot the system. Requires: shutdown_reboot"""
    import subprocess
    import asyncio
    
    # Schedule reboot in 2 seconds to allow response to be sent
    async def do_reboot():
        await asyncio.sleep(2)
        subprocess.run(["sudo", "reboot"], check=False)
    
    asyncio.create_task(do_reboot())
    return {"status": "rebooting", "message": "System will reboot in 2 seconds"}


@api_router.get("/system/health")
async def health_check():
    """Health check endpoint."""
    return {"status": "healthy", "timestamp": datetime.utcnow().isoformat()}


# ============================================
# FRONTEND ERROR LOGGING
# ============================================

class FrontendErrorRequest(BaseModel):
    """Frontend error log request."""
    type: str
    message: str
    stack: Optional[str] = None
    timestamp: str
    url: Optional[str] = None
    userAgent: Optional[str] = None


@api_router.post("/log/frontend-error")
async def log_frontend_error(error: FrontendErrorRequest):
    """
    Log frontend errors for debugging.
    Writes to ~/.seqmaster/logs/frontend.log
    """
    from src.core.config import settings
    import structlog
    
    logger = structlog.get_logger("frontend")
    
    # Ensure logs directory exists
    log_dir = settings.LOGS_DIR
    log_dir.mkdir(parents=True, exist_ok=True)
    log_file = log_dir / "frontend.log"
    
    # Format log entry
    log_entry = f"""
================================================================================
[{error.timestamp}] {error.type}
URL: {error.url or 'N/A'}
User-Agent: {error.userAgent or 'N/A'}
--------------------------------------------------------------------------------
{error.message}
--------------------------------------------------------------------------------
Stack:
{error.stack or 'No stack trace'}
================================================================================
"""
    
    # Append to log file
    try:
        with open(log_file, "a") as f:
            f.write(log_entry)
        
        # Also log to structlog for console visibility
        logger.warning(
            "Frontend error",
            type=error.type,
            message=error.message[:200] if error.message else None,
            url=error.url
        )
    except Exception as e:
        logger.error("Failed to write frontend error log", error=str(e))
    


    return {"status": "logged"}


@api_router.get("/log/frontend-errors")
async def get_frontend_errors(lines: int = 100):
    """
    Get recent frontend errors for debugging.
    Returns last N lines of frontend.log
    """
    from src.core.config import settings
    
    log_file = settings.LOGS_DIR / "frontend.log"
    
    if not log_file.exists():
        return {"errors": [], "message": "No frontend errors logged yet"}
    
    try:
        with open(log_file, "r") as f:
            content = f.read()
        
        # Split by separator and return last N entries
        entries = content.split("=" * 80)
        entries = [e.strip() for e in entries if e.strip()]
        recent = entries[-lines:] if len(entries) > lines else entries
        
        return {
            "total_entries": len(entries),
            "returned": len(recent),
            "errors": recent
        }
    except Exception as e:
        return {"errors": [], "message": f"Error reading log: {str(e)}"}


@api_router.delete("/log/frontend-errors")
async def clear_frontend_errors():
    """Clear frontend error log."""
    from src.core.config import settings
    
    log_file = settings.LOGS_DIR / "frontend.log"
    
    if log_file.exists():
        log_file.unlink()
    
    return {"status": "cleared"}


@api_router.get("/system/storage", response_model=StorageInfoResponse)
async def get_storage_info():
    """Get storage usage breakdown by category."""
    import shutil
    from src.core.config import settings
    
    def get_dir_size(path: Path) -> int:
        """Calculate total size of directory recursively."""
        total = 0
        if path.exists():
            for entry in path.rglob('*'):
                if entry.is_file():
                    try:
                        total += entry.stat().st_size
                    except (OSError, PermissionError):
                        pass
        return total
    
    def get_file_size(path: Path) -> int:
        """Get size of a single file."""
        try:
            return path.stat().st_size if path.exists() else 0
        except (OSError, PermissionError):
            return 0
    
    # Get disk usage for the data directory's filesystem
    data_dir = settings.DATA_DIR
    data_dir.mkdir(parents=True, exist_ok=True)
    
    try:
        disk_usage = shutil.disk_usage(str(data_dir))
        total_bytes = disk_usage.total
        free_bytes = disk_usage.free
        used_bytes = disk_usage.used
    except OSError:
        total_bytes = 0
        free_bytes = 0
        used_bytes = 0
    
    # Calculate sizes for each category
    database_size = get_file_size(settings.DATABASE_PATH)
    logs_size = get_dir_size(settings.LOGS_DIR)
    sequences_size = get_dir_size(settings.SEQUENCES_DIR)
    property_sets_size = get_dir_size(settings.PROPERTY_SETS_DIR)
    
    # Test results are in the database, but we can estimate from results count
    # For now, include in database category
    
    # Calculate "other" data in the .seqmaster directory
    seqmaster_dir = Path.home() / ".seqmaster"
    total_seqmaster = get_dir_size(seqmaster_dir)
    known_size = database_size + logs_size + sequences_size + property_sets_size
    other_size = max(0, total_seqmaster - known_size)
    
    categories = [
        StorageCategory(name="Database", size_bytes=database_size, color="#3B82F6"),  # blue
        StorageCategory(name="Logs", size_bytes=logs_size, color="#F59E0B"),  # amber
        StorageCategory(name="Sequences", size_bytes=sequences_size, color="#10B981"),  # green
        StorageCategory(name="Property Sets", size_bytes=property_sets_size, color="#8B5CF6"),  # purple
        StorageCategory(name="Other", size_bytes=other_size, color="#6B7280"),  # gray
    ]
    
    return StorageInfoResponse(
        total_bytes=total_bytes,
        used_bytes=used_bytes,
        free_bytes=free_bytes,
        categories=categories
    )


# ============================================
# TESTER LOCK ENDPOINTS
# ============================================

@api_router.get("/tester/status")
async def get_tester_lock_status():
    """Get current tester lock status."""
    from src.core.tester_lock import get_tester_lock
    lock = get_tester_lock()
    return lock.get_status()


@api_router.post("/tester/acquire")
async def acquire_tester_lock(request: TesterAcquireRequest):
    """
    Acquire exclusive lock on the tester.
    
    Must be called before starting a test to prevent conflicts.
    Lock expires after 5 minutes without heartbeat.
    """
    from src.core.tester_lock import get_tester_lock
    lock = get_tester_lock()
    
    result = await lock.acquire(
        client_id=request.client_id,
        operator_id=request.operator_id,
        operator_name=request.operator_name,
        force=request.force
    )
    
    if not result.get("acquired"):
        raise HTTPException(
            status_code=status.HTTP_409_CONFLICT,
            detail=result
        )
    
    return result


@api_router.post("/tester/release")
async def release_tester_lock(request: TesterReleaseRequest):
    """Release the tester lock."""
    from src.core.tester_lock import get_tester_lock
    lock = get_tester_lock()
    
    result = await lock.release(
        client_id=request.client_id,
        force=request.force
    )
    
    if not result.get("released"):
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail=result
        )
    
    return result


@api_router.post("/tester/heartbeat")
async def tester_heartbeat(request: TesterHeartbeatRequest):
    """Send heartbeat to keep tester lock alive."""
    from src.core.tester_lock import get_tester_lock
    lock = get_tester_lock()
    
    result = await lock.heartbeat(request.client_id)
    
    if not result.get("success"):
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=result
        )
    
    return result


# ============================================
# SEQUENCE MATCHING ENDPOINT
# ============================================

@api_router.post("/sequences/match", response_model=SequenceMatchResponse)
async def match_sequence_for_dut(
    request: SequenceMatchRequest,
    db = Depends(get_db_session)
):
    """
    Find the best matching sequence for a scanned DUT.
    
    Matching priority:
    0. DUT filter match (all dut_filters must match extracted_variables) - HIGHEST
    1. Exact part number match (dut_partnumber field)
    2. Regex pattern match on barcode (scanned_fields patterns)
    3. Hardware version match (dut_hw_version field)
    4. No match - returns matched=False
    
    If exactly one sequence matches via dut_filters, auto_start=True.
    """
    import re
    
    repo = SequenceRepository(db)
    sequences = await repo.get_all()
    
    barcode = request.barcode
    extracted = request.extracted_variables or {}
    
    # Filter sequences based on status
    # Include drafts only if explicitly requested (admin/engineer)
    if request.include_drafts:
        all_sequences = sequences  # All sequences
    else:
        all_sequences = [s for s in sequences if getattr(s, 'status', 'published') == 'published']
    
    # Deduplicate: keep only one version per sequence_id
    # Priority: published > draft, then newest version
    seq_by_id: Dict[str, Any] = {}
    for s in all_sequences:
        sid = s.sequence_id
        status = getattr(s, 'status', 'published')
        
        if sid not in seq_by_id:
            seq_by_id[sid] = s
        else:
            existing = seq_by_id[sid]
            existing_status = getattr(existing, 'status', 'published')
            
            # Prefer published over draft
            if status == 'published' and existing_status == 'draft':
                seq_by_id[sid] = s
            # If same status, prefer newer version
            elif status == existing_status and s.version > existing.version:
                seq_by_id[sid] = s
    
    eligible_sequences = list(seq_by_id.values())
    
    # Collect all matches with their scores
    all_matches: List[MatchedSequenceInfo] = []
    dut_filter_matches: List[MatchedSequenceInfo] = []
    
    for seq in eligible_sequences:
        # Parse sequence content to get scanned_fields and dut_filters
        try:
            content = parse_sequence_content(seq.content)
        except Exception:
            continue
        
        # Priority 0: DUT filter match (highest priority)
        dut_filters = content.get('dut_filters', {})
        if dut_filters:
            # All filter values must match extracted variables
            all_match = True
            for filter_key, filter_value in dut_filters.items():
                if not filter_value:  # Empty filter value = skip
                    continue
                extracted_value = extracted.get(filter_key, '')
                if str(extracted_value).lower() != str(filter_value).lower():
                    all_match = False
                    break
            
            if all_match and any(v for v in dut_filters.values()):  # At least one non-empty filter
                match_info = MatchedSequenceInfo(
                    sequence_id=seq.sequence_id,
                    sequence_name=seq.name,
                    version=seq.version,
                    match_type="dut_filter",
                    confidence=100
                )
                dut_filter_matches.append(match_info)
                all_matches.append(match_info)
                continue  # Don't check other methods for this sequence
        
        # Priority 1: Exact part number match
        if seq.dut_partnumber and extracted.get('Part_Number'):
            if seq.dut_partnumber.lower() == extracted['Part_Number'].lower():
                all_matches.append(MatchedSequenceInfo(
                    sequence_id=seq.sequence_id,
                    sequence_name=seq.name,
                    version=seq.version,
                    match_type="exact_partnumber",
                    confidence=100
                ))
                continue
        
        # Pattern matching via scanned_fields is NOT used for sequence matching
        # It's only used for extracting data from barcodes
        # Matching must be done via dut_filters (exact match)
        
        # Priority 2: Hardware version match (legacy support)
        if seq.dut_hw_version and extracted.get('HW_Version'):
            if seq.dut_hw_version.lower() == extracted['HW_Version'].lower():
                all_matches.append(MatchedSequenceInfo(
                    sequence_id=seq.sequence_id,
                    sequence_name=seq.name,
                    version=seq.version,
                    match_type="hw_version",
                    confidence=80
                ))
    
    # If we have DUT filter matches, only return those (highest priority)
    if dut_filter_matches:
        auto_start = len(dut_filter_matches) == 1
        best = dut_filter_matches[0]
        return SequenceMatchResponse(
            matched=True,
            sequence_id=best.sequence_id,
            sequence_name=best.sequence_name,
            version=best.version,
            match_type="dut_filter",
            confidence=100,
            message=f"Matched {len(dut_filter_matches)} sequence(s) via DUT filters",
            auto_start=auto_start,
            matching_sequences=dut_filter_matches
        )
    
    # Sort other matches by confidence
    all_matches.sort(key=lambda m: m.confidence, reverse=True)
    
    if all_matches:
        best = all_matches[0]
        return SequenceMatchResponse(
            matched=True,
            sequence_id=best.sequence_id,
            sequence_name=best.sequence_name,
            version=best.version,
            match_type=best.match_type,
            confidence=best.confidence,
            message=f"Matched via {best.match_type}",
            auto_start=False,  # Only auto_start for dut_filter matches
            matching_sequences=all_matches
        )
    
    return SequenceMatchResponse(
        matched=False,
        match_type="none",
        confidence=0,
        message="No matching sequence found. Please select manually.",
        auto_start=False,
        matching_sequences=[]
    )


# ============================================
# SEQUENCE ENDPOINTS
# ============================================

@api_router.get("/sequences")
async def list_sequences(db = Depends(get_db_session)):
    """List all sequences."""
    repo = SequenceRepository(db)
    sequences = await repo.get_all()
    
    result = []
    for s in sequences:
        seq_data = {
            "id": s.id,
            "sequence_id": s.sequence_id,
            "version": s.version,
            "name": s.name,
            "description": s.description,
            "status": getattr(s, 'status', 'published'),
            "dut_description": getattr(s, 'dut_description', None),
            "created_at": s.created_at.isoformat()
        }
        
        # Parse content to get scanned_fields and dut_filters
        if s.content:
            try:
                content_data = parse_sequence_content(s.content)
                fields = content_data.get('scanned_fields')
                if fields:
                    seq_data['scanned_fields'] = fields
                filters = content_data.get('dut_filters')
                if filters:
                    seq_data['dut_filters'] = filters
            except Exception:
                pass  # Skip if content can't be parsed
        
        result.append(seq_data)
    
    return result


@api_router.get("/sequences/{sequence_id}")
async def get_sequence(sequence_id: str, version: Optional[str] = None, 
                       db = Depends(get_db_session)):
    """Get a sequence by ID."""
    repo = SequenceRepository(db)
    
    if version:
        sequence = await repo.get_by_version(sequence_id, version)
    else:
        sequence = await repo.get_latest(sequence_id)
    
    if not sequence:
        raise HTTPException(status_code=404, detail="Sequence not found")
    
    return {
        "id": sequence.id,
        "sequence_id": sequence.sequence_id,
        "version": sequence.version,
        "name": sequence.name,
        "content": sequence.content,
        "content_hash": sequence.content_hash,
        "status": sequence.status,
        "dut_description": sequence.dut_description,
        "dut_hw_version": sequence.dut_hw_version,
        "dut_partnumber": sequence.dut_partnumber,
        "dut_batch_number": sequence.dut_batch_number,
        "created_at": sequence.created_at.isoformat(),
        "published_at": sequence.published_at.isoformat() if sequence.published_at else None
    }


@api_router.get("/sequences/{sequence_id}/versions")
async def get_sequence_versions(sequence_id: str, db = Depends(get_db_session)):
    """Get all versions of a sequence."""
    repo = SequenceRepository(db)
    versions = await repo.get_versions(sequence_id)
    
    if not versions:
        raise HTTPException(status_code=404, detail="Sequence not found")
    
    return [
        {
            "id": v.id,
            "version": v.version,
            "status": v.status,
            "created_at": v.created_at.isoformat(),
            "published_at": v.published_at.isoformat() if v.published_at else None
        }
        for v in versions
    ]


@api_router.post("/sequences/{sequence_id}/validate")
async def validate_sequence(sequence_id: str, 
                           version: Optional[str] = None,
                           db = Depends(get_db_session)):
    """
    Validate a sequence for errors before execution.
    
    Checks:
    - Parameter types (numbers vs strings)
    - Required fields
    - Valid step types
    - Expression syntax
    - Limit configurations
    """
    import json
    import yaml
    
    repo = SequenceRepository(db)
    
    if version:
        sequence = await repo.get_by_version(sequence_id, version)
    else:
        sequence = await repo.get_latest(sequence_id)
    
    if not sequence:
        raise HTTPException(status_code=404, detail="Sequence not found")
    
    # Parse sequence
    data = parse_sequence_content(sequence.content)
    
    errors = []
    warnings = []
    
    # Validate each group and step
    for group_idx, group in enumerate(data.get("groups", [])):
        group_name = group.get("name", f"Group {group_idx + 1}")
        
        for step_idx, step in enumerate(group.get("steps", [])):
            step_id = step.get("id", f"Step {step_idx + 1}")
            step_name = step.get("name", step_id)
            step_type = step.get("type", "action")
            params = step.get("parameters", {})
            
            # Check numeric parameters that might be strings
            numeric_params = ["delay_ms", "milliseconds", "seconds", "timeout_ms", 
                            "lower_limit", "upper_limit", "nominal", "tolerance",
                            "count", "retries"]
            
            for param in numeric_params:
                if param in params:
                    value = params[param]
                    if isinstance(value, str):
                        # Check if it's a valid number string
                        try:
                            float(value)
                            warnings.append({
                                "step_id": step_id,
                                "step_name": step_name,
                                "group": group_name,
                                "type": "warning",
                                "code": "STRING_NUMBER",
                                "message": f"Parameter '{param}' is a string ('{value}') but should be a number. Will auto-convert.",
                                "field": param,
                                "value": value
                            })
                        except ValueError:
                            # Not a valid number - check if it's an expression
                            if not ("$" in value or "Locals." in value or "Parameters." in value):
                                errors.append({
                                    "step_id": step_id,
                                    "step_name": step_name,
                                    "group": group_name,
                                    "type": "error",
                                    "code": "INVALID_NUMBER",
                                    "message": f"Parameter '{param}' has invalid value '{value}' - expected a number",
                                    "field": param,
                                    "value": value
                                })
            
            # Check limits configuration
            limits = step.get("limits", {})
            if limits:
                for limit_field in ["lower", "upper", "nominal", "tolerance"]:
                    if limit_field in limits:
                        value = limits[limit_field]
                        if isinstance(value, str) and not ("$" in value or "." in value and any(x in value for x in ["Locals", "Parameters", "Station"])):
                            try:
                                float(value)
                                warnings.append({
                                    "step_id": step_id,
                                    "step_name": step_name,
                                    "group": group_name,
                                    "type": "warning",
                                    "code": "STRING_LIMIT",
                                    "message": f"Limit '{limit_field}' is a string ('{value}') but should be a number",
                                    "field": f"limits.{limit_field}",
                                    "value": value
                                })
                            except ValueError:
                                errors.append({
                                    "step_id": step_id,
                                    "step_name": step_name,
                                    "group": group_name,
                                    "type": "error",
                                    "code": "INVALID_LIMIT",
                                    "message": f"Limit '{limit_field}' has invalid value '{value}'",
                                    "field": f"limits.{limit_field}",
                                    "value": value
                                })
            
            # Check required fields
            if not step.get("id"):
                errors.append({
                    "step_id": f"step_{group_idx}_{step_idx}",
                    "step_name": step_name,
                    "group": group_name,
                    "type": "error",
                    "code": "MISSING_ID",
                    "message": "Step is missing required 'id' field"
                })
            
            if not step.get("name"):
                warnings.append({
                    "step_id": step_id,
                    "step_name": "(unnamed)",
                    "group": group_name,
                    "type": "warning",
                    "code": "MISSING_NAME",
                    "message": "Step is missing 'name' field - will use id as name"
                })
    
    # ============================
    # CHECK: Duplicate step IDs
    # ============================
    all_step_ids = []
    for group in data.get("groups", []):
        for step in group.get("steps", []):
            step_id = step.get("id")
            if step_id:
                all_step_ids.append({
                    "id": step_id,
                    "name": step.get("name", step_id),
                    "group": group.get("name", "Unknown")
                })
    
    # Find duplicates
    seen_ids = {}
    for step_info in all_step_ids:
        step_id = step_info["id"]
        if step_id in seen_ids:
            errors.append({
                "step_id": step_id,
                "step_name": step_info["name"],
                "group": step_info["group"],
                "type": "error",
                "code": "DUPLICATE_ID",
                "message": f"Duplicate step ID '{step_id}' - also used in group '{seen_ids[step_id]}'"
            })
        else:
            seen_ids[step_id] = step_info["group"]
    
    # ============================
    # CHECK: Undefined variables
    # ============================
    import re
    
    # Get defined variables
    defined_vars = set()
    for scope in ["Locals", "Parameters", "Station"]:
        scope_vars = data.get("variables", {}).get(scope, {})
        for var_name in scope_vars.keys():
            defined_vars.add(f"{scope}.{var_name}")
    
    # Find all variable references in the sequence
    variable_pattern = re.compile(r'\b(Locals|Parameters|Station)\.([a-zA-Z_][a-zA-Z0-9_]*)')
    
    for group in data.get("groups", []):
        group_name = group.get("name", "Unknown")
        for step in group.get("steps", []):
            step_id = step.get("id", "unknown")
            step_name = step.get("name", step_id)
            
            # Convert step to string to find all variable references
            step_str = str(step)
            
            # Find all variable references
            matches = variable_pattern.findall(step_str)
            for scope, var_name in matches:
                var_ref = f"{scope}.{var_name}"
                # Get root variable (before any dots for nested properties)
                root_var = f"{scope}.{var_name.split('.')[0]}"
                
                if root_var not in defined_vars:
                    warnings.append({
                        "step_id": step_id,
                        "step_name": step_name,
                        "group": group_name,
                        "type": "warning",
                        "code": "UNDEFINED_VARIABLE",
                        "message": f"Variable '{var_ref}' is used but not defined in variables section",
                        "field": "variable",
                        "value": var_ref
                    })
                    # Add to defined_vars to avoid duplicate warnings
                    defined_vars.add(root_var)
    
    return {
        "sequence_id": sequence_id,
        "version": sequence.version,
        "valid": len(errors) == 0,
        "error_count": len(errors),
        "warning_count": len(warnings),
        "errors": errors,
        "warnings": warnings
    }


@api_router.post("/sequences")
async def upload_sequence(request: SequenceUploadRequest,
                         user = Depends(require_permission("create_sequences")),
                         db = Depends(get_db_session)):
    """Upload a new sequence version. Requires: create_sequences"""
    repo = SequenceRepository(db)
    
    # ── Demo mode enforcement: max sequences ──
    license_svc = get_license_service()
    if license_svc.is_demo():
        existing = await repo.get_all()
        unique_ids = set(s.sequence_id for s in existing)
        max_seq = DEMO_LIMITS.get("max_sequences", 3)
        if request.sequence_id not in unique_ids and len(unique_ids) >= max_seq:
            raise HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=f"Demo mode: max {max_seq} sekvenser. Opgrader licens for ubegrænset."
            )
    
    sequence = await repo.create(
        sequence_id=request.sequence_id,
        version=request.version,
        name=request.name,
        content=request.content,
        dut_description=request.dut_type,
        description=request.description,
        created_by="admin"  # Hardcoded for MVP
    )
    
    await db.commit()
    
    return {
        "id": sequence.id,
        "sequence_id": sequence.sequence_id,
        "version": sequence.version,
        "content_hash": sequence.content_hash
    }


@api_router.post("/sequences/upload")
async def upload_sequence_file(file: UploadFile,
                               user = Depends(require_permission("create_sequences")),
                               db = Depends(get_db_session)):
    """Upload a sequence from JSON/YAML file. Requires: create_sequences"""
    import json
    import yaml
    from hashlib import sha256
    
    # Read file content
    content = await file.read()
    content_text = content.decode('utf-8')
    
    # Parse content (auto-detects JSON or YAML)
    data = parse_sequence_content(content_text)
    
    repo = SequenceRepository(db)
    
    # Use 'id' field from JSON as sequence_id
    sequence_id = data.get('id') or data.get('sequence_id', 'SEQ-' + sha256(content).hexdigest()[:8].upper())
    
    # ── Demo mode enforcement: max sequences ──
    license_svc = get_license_service()
    if license_svc.is_demo():
        existing = await repo.get_all()
        unique_ids = set(s.sequence_id for s in existing)
        max_seq = DEMO_LIMITS.get("max_sequences", 3)
        if sequence_id not in unique_ids and len(unique_ids) >= max_seq:
            raise HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=f"Demo mode: max {max_seq} sekvenser. Opgrader licens for ubegrænset."
            )
    
    sequence = await repo.create(
        sequence_id=sequence_id,
        version=data.get('version', '1.0'),
        name=data.get('name', file.filename),
        content=content_text,
        dut_type=data.get('dut_type'),
        description=data.get('description'),
        created_by="admin"
    )
    
    await db.commit()
    
    return {
        "id": sequence.id,
        "sequence_id": sequence.sequence_id,
        "version": sequence.version,
        "content_hash": sequence.content_hash
    }


# Note: get_sequence_versions is already defined earlier (line ~212) with more fields
# This duplicate was removed to avoid FastAPI route conflict


@api_router.post("/sequences/{sequence_id}/draft")
async def save_draft(sequence_id: str, request: Dict[str, Any],
                    user = Depends(require_permission("edit_sequences")),
                    db = Depends(get_db_session)):
    """Save sequence as draft.
    
    Draft handling:
    - Only ONE draft can exist per sequence_id at a time
    - When editing a published version, its content is copied to the draft
    - The draft's version indicates which published version it's based on
    - Published versions are immutable (never modified)
    
    Requires: edit_sequences
    """
    from hashlib import sha256
    import json
    
    repo = SequenceRepository(db)
    content = json.dumps(request.get("content"))
    
    # Prepare version string (add -draft if not already present)
    base_version = request.get("version", "1.0")
    if not base_version.endswith("-draft"):
        draft_version = base_version + "-draft"
    else:
        draft_version = base_version
    
    # Check if there's an existing draft for this sequence
    existing_draft = await repo.get_draft(sequence_id)
    
    if existing_draft:
        # Update existing draft with new content
        existing_draft.name = request.get("name", existing_draft.name)
        existing_draft.description = request.get("description")
        existing_draft.content = content
        existing_draft.content_hash = sha256(content.encode()).hexdigest()
        existing_draft.version = draft_version
        existing_draft.dut_description = request.get("dut_description")
        existing_draft.dut_hw_version = request.get("dut_hw_version")
        existing_draft.dut_partnumber = request.get("dut_partnumber")
        existing_draft.dut_batch_number = request.get("dut_batch_number")
        sequence = existing_draft
    else:
        # Create new draft
        sequence = await repo.create(
            sequence_id=sequence_id,
            version=draft_version,
            name=request.get("name"),
            content=content,
            description=request.get("description"),
            dut_description=request.get("dut_description"),
            dut_hw_version=request.get("dut_hw_version"),
            dut_partnumber=request.get("dut_partnumber"),
            dut_batch_number=request.get("dut_batch_number"),
            status="draft",
            created_by="admin"
        )
    
    await db.commit()
    
    return {
        "id": sequence.id,
        "sequence_id": sequence.sequence_id,
        "version": sequence.version,
        "status": sequence.status
    }


@api_router.post("/sequences/{sequence_id}/publish")
async def publish_sequence(sequence_id: str,
                          user = Depends(require_permission("publish_sequences")),
                          db = Depends(get_db_session)):
    """Publish draft as new version. Creates a NEW record for the published version.
    
    This ensures version immutability - each published version is a separate
    database record that cannot be modified. The draft remains for continued editing.
    
    Requires: publish_sequences
    """
    from datetime import datetime
    from hashlib import sha256
    
    repo = SequenceRepository(db)
    
    # Get draft - use get_draft() which specifically looks for status='draft'
    draft = await repo.get_draft(sequence_id)
    if not draft:
        raise HTTPException(status_code=404, detail="No draft found to publish")
    
    # Get all published versions to determine next version
    versions = await repo.get_versions(sequence_id)
    published_versions = [v for v in versions if v.status == "published"]
    
    # Calculate next version number
    if published_versions:
        # Find the highest version number
        def version_key(v):
            try:
                parts = v.version.split('.')
                return (int(parts[0]), int(parts[1]))
            except (ValueError, IndexError):
                return (0, 0)
        latest = max(published_versions, key=version_key)
        try:
            major, minor = latest.version.split('.')
            next_version = f"{major}.{int(minor) + 1}"
        except (ValueError, IndexError):
            next_version = "1.0"
    else:
        next_version = "1.0"
    
    # CREATE A NEW RECORD for the published version (immutable)
    # This is the key fix - we copy the content to a new record instead
    # of modifying the existing draft
    published = await repo.create(
        sequence_id=sequence_id,
        version=next_version,
        name=draft.name,
        content=draft.content,
        description=draft.description,
        dut_description=draft.dut_description,
        dut_hw_version=draft.dut_hw_version,
        dut_partnumber=draft.dut_partnumber,
        dut_batch_number=draft.dut_batch_number,
        status="published",
        created_by=draft.created_by
    )
    published.published_at = datetime.utcnow()
    
    # Keep the draft for continued editing, but update its base version
    # Remove -draft suffix for display, the save_draft endpoint will add it back
    draft.version = next_version + "-draft"
    
    await db.commit()
    
    return {
        "id": published.id,
        "sequence_id": published.sequence_id,
        "version": published.version,
        "status": published.status,
        "published_at": published.published_at.isoformat()
    }


@api_router.delete("/sequences/{sequence_id}/draft")
async def delete_draft(sequence_id: str,
                      user = Depends(require_permission("edit_sequences")),
                      db = Depends(get_db_session)):
    """Delete only the draft version of a sequence. Requires: edit_sequences"""
    repo = SequenceRepository(db)
    
    # Get draft version
    draft = await repo.get_latest(sequence_id)
    if not draft or draft.status != "draft":
        raise HTTPException(status_code=404, detail="No draft found for this sequence")
    
    # Delete draft
    await db.delete(draft)
    await db.commit()
    
    return {"message": f"Deleted draft for sequence {sequence_id}"}


@api_router.delete("/sequences/{sequence_id}/versions/{version}")
async def delete_sequence_version(sequence_id: str,
                                  version: str,
                                  user = Depends(require_permission("delete_sequences")),
                                  db = Depends(get_db_session)):
    """Delete a specific version of a sequence. Requires: delete_sequences"""
    repo = SequenceRepository(db)
    
    # Get specific version
    sequence = await repo.get_by_version(sequence_id, version)
    if not sequence:
        raise HTTPException(status_code=404, detail=f"Sequence {sequence_id} version {version} not found")
    
    # Delete this version
    await db.delete(sequence)
    await db.commit()
    
    return {"message": f"Deleted sequence {sequence_id} version {version}"}


# ============================================
# PROPERTY SET ENDPOINTS
# ============================================

@api_router.get("/property-sets")
async def list_property_sets(db = Depends(get_db_session)):
    """List all property sets."""
    repo = PropertySetRepository(db)
    property_sets = await repo.get_all()
    return [
        {
            "id": ps.id,
            "property_set_id": ps.property_set_id,
            "version": ps.version,
            "name": ps.name,
            "dut_type": ps.dut_type,
            "dut_revision": ps.dut_revision,
            "created_at": ps.created_at.isoformat() if ps.created_at else None
        }
        for ps in property_sets
    ]


@api_router.post("/property-sets")
async def upload_property_set(request: PropertySetUploadRequest,
                              user = Depends(require_permission("manage_property_sets")),
                              db = Depends(get_db_session)):
    """Upload a new property set version. Requires: manage_property_sets"""
    repo = PropertySetRepository(db)
    
    prop_set = await repo.create(
        property_set_id=request.property_set_id,
        version=request.version,
        name=request.name,
        content=request.content,
        dut_type=request.dut_type,
        dut_revision=request.dut_revision,
        created_by=user.username if user else None
    )
    
    await db.commit()
    
    return {
        "id": prop_set.id,
        "property_set_id": prop_set.property_set_id,
        "version": prop_set.version,
        "content_hash": prop_set.content_hash
    }


# ============================================
# EXECUTION ENDPOINTS
# ============================================

@api_router.post("/execution/start")
async def start_execution(request: ExecutionStartRequest,
                         user = Depends(require_permission("run_tests")),
                         db = Depends(get_db_session)):
    """Start test execution. Requires: run_tests"""
    
    # ── Demo mode enforcement: max stored results ──
    license_svc = get_license_service()
    if license_svc.is_demo():
        session_repo_check = TestSessionRepository(db)
        recent = await session_repo_check.get_recent(limit=9999)
        max_results = DEMO_LIMITS.get("max_results", 10)
        if len(recent) >= max_results:
            raise HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=f"Demo mode: max {max_results} gemte resultater. "
                       f"Slet gamle resultater eller opgrader licens."
            )
    
    from src.executor.engine import SequenceExecutor
    from src.executor.models import TestSequence
    import json
    import yaml
    
    # Get sequence
    seq_repo = SequenceRepository(db)
    if request.sequence_version:
        sequence_record = await seq_repo.get_by_version(
            request.sequence_id, request.sequence_version
        )
    else:
        sequence_record = await seq_repo.get_latest(request.sequence_id)
    
    if not sequence_record:
        raise HTTPException(status_code=404, detail="Sequence not found")
    
    # Parse sequence (YAML or JSON)
    sequence_data = parse_sequence_content(sequence_record.content)
    sequence = TestSequence(**sequence_data)
    
    # Get property set if specified
    property_set = None
    if request.property_set_id:
        prop_repo = PropertySetRepository(db)
        prop_record = await prop_repo.get_latest(request.property_set_id)
        if prop_record:
            from src.executor.models import PropertySet
            prop_data = json.loads(prop_record.content)
            property_set = PropertySet(**prop_data)
    
    # Get or create executor
    executor = ServiceRegistry.get("executor")
    if not executor:
        from src.drivers.manager import get_driver_manager
        watchdog = ServiceRegistry.get("watchdog")
        executor = SequenceExecutor(
            driver_manager=get_driver_manager(),
            watchdog=watchdog
        )
        ServiceRegistry.set("executor", executor)
    
    # Set WebSocket callbacks for real-time updates
    ws_broadcast_status = ServiceRegistry.get("ws_broadcast_status")
    ws_broadcast_step = ServiceRegistry.get("ws_broadcast_step")
    ws_broadcast_operator = ServiceRegistry.get("ws_broadcast_operator_input")
    executor.set_callbacks(
        on_status_update=ws_broadcast_status,
        on_step_complete=ws_broadcast_step,
        on_operator_input=ws_broadcast_operator
    )
    
    # Create session in database
    session_repo = TestSessionRepository(db)
    session_id = str(__import__('uuid').uuid4())
    session = await session_repo.create(
        session_id=session_id,
        sequence_id=sequence_record.id,  # Use database ID, not sequence_id string
        sequence_version=sequence_record.version,
        dut_id=request.dut_id,
        dut_serial=request.dut_serial,
        operator=user.get("username", "operator") if isinstance(user, dict) else "operator",
        property_set_id=prop_record.id if request.property_set_id and prop_record else None
    )
    await db.commit()
    
    # Start execution in background
    import asyncio
    
    # Capture scanned_fields and operator for closure
    scanned = request.scanned_fields
    operator_name = user.get("username", "operator") if isinstance(user, dict) else "operator"
    
    async def run_execution():
        from src.database.connection import async_session_maker
        from src.database.models import StepResult, StepStatus
        
        try:
            context = await executor.execute(
                sequence=sequence,
                dut_id=request.dut_id,
                operator=operator_name,
                property_set=property_set,
                dry_run=request.dry_run,
                session_id=session_id,  # Pass session_id to executor
                scanned_fields=scanned  # Pass scanned fields from barcode
            )
            
            # Update session with results - use new DB session
            async with async_session_maker() as new_db:
                session_repo = TestSessionRepository(new_db)
                session_obj = await session_repo.get_by_session_id(session_id)
                
                if session_obj:
                    session_obj.completed_at = context.completed_at
                    session_obj.passed_steps = context.passed_steps
                    session_obj.failed_steps = context.failed_steps
                    session_obj.total_steps = context.total_steps
                    
                    if context.failed_steps == 0:
                        session_obj.status = TestStatus.PASSED
                    else:
                        session_obj.status = TestStatus.FAILED
                    
                    def _coerce_float(value):
                        if isinstance(value, (int, float)):
                            return float(value)
                        if isinstance(value, str):
                            try:
                                return float(value.strip())
                            except ValueError:
                                return None
                        return None

                    # Save step results
                    for idx, step_result in enumerate(context.step_results):
                        measured_value = _coerce_float(step_result.measured_value)
                        expected_value = _coerce_float(step_result.expected_value)
                        measured_value_str = str(step_result.measured_value) if step_result.measured_value is not None else None
                        expected_value_str = str(step_result.expected_value) if step_result.expected_value is not None else None
                        # Map executor status to database status
                        db_status = StepStatus.PENDING
                        if step_result.status:
                            status_map = {
                                'passed': StepStatus.PASSED,
                                'failed': StepStatus.FAILED,
                                'skipped': StepStatus.SKIPPED,
                                'error': StepStatus.ERROR,
                                'running': StepStatus.RUNNING,
                                'executed': StepStatus.EXECUTED,  # Flow control steps
                            }
                            db_status = status_map.get(step_result.status.value, StepStatus.PENDING)
                        
                        step_record = StepResult(
                            session_id=session_obj.id,
                            step_id=step_result.step_id,
                            step_index=idx,
                            step_name=step_result.step_name,
                            step_type=step_result.step_type,
                            group_id=step_result.group_id,
                            group_name=step_result.group_name,
                            adapter=getattr(step_result, 'adapter', None),
                            comparison_operator=getattr(step_result, 'comparison_operator', None),
                            started_at=step_result.started_at,
                            completed_at=step_result.completed_at,
                            duration_ms=step_result.duration_ms,
                            status=db_status,
                            measured_value=measured_value,
                            measured_value_str=measured_value_str,
                            expected_value=expected_value,
                            expected_value_str=expected_value_str,
                            lower_limit=step_result.lower_limit,
                            upper_limit=step_result.upper_limit,
                            passed=step_result.passed,
                            error_code=step_result.error_code,
                            error_message=step_result.error_message,
                            retry_count=step_result.retry_count
                        )
                        new_db.add(step_record)
                    
                    await new_db.commit()
                    
        except Exception as e:
            import structlog
            logger = structlog.get_logger(__name__)
            logger.error("Execution failed", error=str(e), session_id=session_id)
            
            # Update session with error - use new DB session
            async with async_session_maker() as new_db:
                session_repo = TestSessionRepository(new_db)
                session_obj = await session_repo.get_by_session_id(session_id)
                
                if session_obj:
                    session_obj.status = TestStatus.FAILED
                    session_obj.error_message = str(e)
                    await new_db.commit()
    
    asyncio.create_task(run_execution())
    
    return {
        "status": "started",
        "session_id": session_id
    }


@api_router.post("/execution/control")
async def control_execution(request: ExecutionControlRequest,
                           user = Depends(require_permission("abort_tests"))):
    """Control running execution (pause/resume/abort). Requires: abort_tests"""
    executor = ServiceRegistry.get("executor")
    
    if not executor:
        raise HTTPException(status_code=400, detail="No active execution")
    
    if request.action == "pause":
        await executor.pause()
    elif request.action == "resume":
        await executor.resume()
    elif request.action == "abort":
        await executor.abort()
    else:
        raise HTTPException(status_code=400, detail=f"Unknown action: {request.action}")
    
    return {"status": executor.state.value}


@api_router.get("/execution/status")
async def get_execution_status(
    include_steps: bool = False,
    step_limit: Optional[int] = None
):
    """Get current execution status including step results for real-time display."""
    executor = ServiceRegistry.get("executor")
    
    if not executor or not executor.context:
        return {
            "is_running": False,
            "state": "idle"
        }
    
    ctx = executor.context
    status = executor._get_status()
    status["is_running"] = executor.is_running()
    status["is_paused"] = executor.is_paused()
    status["progress"] = ctx.current_step_index
    status["total_steps"] = ctx.total_steps
    status["executed_steps"] = ctx.executed_steps
    status["passed_steps"] = ctx.passed_steps
    status["failed_steps"] = ctx.failed_steps
    status["skipped_steps"] = getattr(ctx, 'skipped_steps', 0)
    status["current_step"] = status.get("current_step_name")
    status["current_group"] = ctx.current_group.name if ctx.current_group else None
    
    # Loop info
    status["loop_iteration"] = ctx.current_loop_iteration if ctx.current_loop_total > 0 else None
    status["loop_total"] = ctx.current_loop_total if ctx.current_loop_total > 0 else None
    
    # Flow step counter
    status["flow_steps"] = ctx.flow_steps
    
    # Include step results for real-time step log
    if include_steps:
        step_results = ctx.step_results
        if step_limit:
            step_results = step_results[-step_limit:]
        status["step_results"] = [serialize_step_result(r) for r in step_results]
    else:
        status["step_results"] = []
    
    return status


@api_router.post("/execution/operator-input")
async def submit_operator_input(request: OperatorInputRequest,
                                user = Depends(get_current_user)):
    """Submit operator input for a pending step."""
    # Handle operator input
    return {"status": "accepted"}


# ============================================
# TEST RESULTS ENDPOINTS
# ============================================

@api_router.get("/results")
async def list_results(limit: int = 50, dut_id: Optional[str] = None,
                       db = Depends(get_db_session)):
    """List test results."""
    repo = TestSessionRepository(db)
    
    if dut_id:
        sessions = await repo.get_by_dut(dut_id, limit=limit)
    else:
        sessions = await repo.get_recent(limit=limit)
    
    return [session_summary(s) for s in sessions]


@api_router.get("/results/{session_id}")
async def get_result(
    session_id: str,
    include_steps: bool = False,
    step_limit: Optional[int] = None,
    db = Depends(get_db_session)
):
    """Get detailed test result."""
    session_repo = TestSessionRepository(db)
    step_repo = StepResultRepository(db)
    
    session = await session_repo.get_by_session_id(session_id)
    if not session:
        raise HTTPException(status_code=404, detail="Session not found")
    
    steps = []
    if include_steps:
        if step_limit:
            steps = await step_repo.get_by_session(session.id, limit=step_limit, newest_first=True)
            steps = list(reversed(steps))
        else:
            steps = await step_repo.get_by_session(session.id)
    
    # Calculate pass status
    passed = session.failed_steps == 0 if session.failed_steps is not None else None
    
    # Calculate duration
    duration = duration_seconds(session.started_at, session.completed_at)
    
    base = session_summary(session)
    base.update({
        "pass": passed,
        "property_set_version": session.property_set_version,
        "error_code": session.error_code,
        "error_message": session.error_message,
        "steps": [serialize_step_result(s) for s in steps]
    })
    return base


@api_router.delete("/results/{session_id}")
async def delete_result(session_id: str,
                       user = Depends(require_permission("delete_results")),
                       db = Depends(get_db_session)):
    """Delete a test result. Requires: delete_results"""
    session_repo = TestSessionRepository(db)
    step_repo = StepResultRepository(db)
    
    session = await session_repo.get_by_session_id(session_id)
    if not session:
        raise HTTPException(status_code=404, detail="Session not found")
    
    # Delete step results first (bulk)
    await db.execute(
        delete(StepResult).where(StepResult.session_id == session.id)
    )
    
    # Delete session
    await db.delete(session)
    await db.commit()
    
    return {"message": f"Deleted result {session_id}"}


@api_router.get("/results/{session_id}/export")
async def export_result(session_id: str, format: str = "json",
                        user = Depends(require_permission("export_results")),
                        db = Depends(get_db_session)):
    """Export test result in various formats. Requires: export_results"""
    # Get result data
    result = await get_result(session_id, include_steps=True, db=db)
    
    if format == "json":
        return result
    elif format == "csv":
        # Generate CSV
        import csv
        import io
        
        output = io.StringIO()
        writer = csv.writer(output)
        
        # Header
        writer.writerow(["Step ID", "Step Name", "Status", "Measured", 
                        "Lower Limit", "Upper Limit", "Passed", "Duration (ms)"])
        
        for step in result["steps"]:
            writer.writerow([
                step["step_id"],
                step["step_name"],
                step["status"],
                step["measured_value"],
                step["lower_limit"],
                step["upper_limit"],
                step["passed"],
                step["duration_ms"]
            ])
        
        from fastapi.responses import Response
        return Response(
            content=output.getvalue(),
            media_type="text/csv",
            headers={"Content-Disposition": f"attachment; filename={session_id}.csv"}
        )
    else:
        raise HTTPException(status_code=400, detail=f"Unknown format: {format}")


# ============================================
# DRIVERS ENDPOINTS
# ============================================

@api_router.get("/drivers")
async def list_drivers():
    """List all registered drivers."""
    from src.drivers.manager import get_driver_manager
    manager = get_driver_manager()
    return manager.get_status()


@api_router.post("/drivers/{device_id}/connect")
async def connect_driver(device_id: str, 
                        user = Depends(require_permission("configure_hardware"))):
    """Connect a driver. Requires: configure_hardware"""
    from src.drivers.manager import get_driver_manager
    manager = get_driver_manager()
    
    driver = manager.get_driver(device_id)
    if not driver:
        raise HTTPException(status_code=404, detail="Driver not found")
    
    result = await driver.connect()
    return {"success": result.success, "error": result.error}


@api_router.post("/drivers/{device_id}/disconnect")
async def disconnect_driver(device_id: str,
                           user = Depends(require_permission("configure_hardware"))):
    """Disconnect a driver. Requires: configure_hardware"""
    from src.drivers.manager import get_driver_manager
    manager = get_driver_manager()
    
    driver = manager.get_driver(device_id)
    if not driver:
        raise HTTPException(status_code=404, detail="Driver not found")
    
    result = await driver.disconnect()
    return {"success": result.success}


# ============================================
# SYNC ENDPOINTS (Option D Architecture)
# ============================================

@api_router.get("/sync/status")
async def get_sync_status():
    """Get sync service status."""
    sync_service = ServiceRegistry.get("sync_service")
    if not sync_service:
        return {"enabled": False, "message": "Sync service not available"}
    
    return await sync_service.get_status()


@api_router.post("/sync/force")
async def force_sync():
    """Force immediate sync with Central."""
    sync_service = ServiceRegistry.get("sync_service")
    if not sync_service:
        raise HTTPException(status_code=503, detail="Sync service not available")
    
    return await sync_service.force_sync()


@api_router.get("/sync/queue")
async def get_sync_queue():
    """Get pending sync queue items."""
    sync_service = ServiceRegistry.get("sync_service")
    if not sync_service:
        return {"pending": 0}
    
    stats = await sync_service.queue.get_queue_stats()
    return {"queue_stats": stats}

