"""
SeqMaster Runtime - API Serializers
Layer: API

Shared serialization helpers for API responses.
"""

from datetime import datetime
from typing import Any, Dict, Optional


def isoformat_z(value: Optional[datetime]) -> Optional[str]:
    """Return ISO-8601 string with Z suffix or None."""
    return (value.isoformat() + "Z") if value else None


def duration_seconds(started_at: Optional[datetime], completed_at: Optional[datetime]) -> Optional[float]:
    """Calculate duration in seconds or None."""
    if started_at and completed_at:
        return (completed_at - started_at).total_seconds()
    return None


def session_summary(session: Any) -> Dict[str, Any]:
    """Serialize common TestSession fields for list results."""
    return {
        "session_id": session.session_id,
        "dut_id": session.dut_id,
        "dut_serial": session.dut_serial,
        "sequence_id": session.sequence.sequence_id if session.sequence else None,
        "sequence_name": session.sequence.name if session.sequence else None,
        "sequence_version": session.sequence_version,
        "status": session.status.value if session.status else None,
        "started_at": isoformat_z(session.started_at),
        "completed_at": isoformat_z(session.completed_at),
        "duration": duration_seconds(session.started_at, session.completed_at),
        "operator": session.operator,
        "passed_steps": session.passed_steps,
        "failed_steps": session.failed_steps,
        "total_steps": session.total_steps
    }


def serialize_step_result(step: Any) -> Dict[str, Any]:
    """Serialize a StepResult or StepExecutionResult consistently."""
    status_value = step.status.value if getattr(step, "status", None) else None
    measured = step.measured_value if getattr(step, "measured_value", None) is not None else getattr(step, "measured_value_str", None)
    expected = step.expected_value if getattr(step, "expected_value", None) is not None else getattr(step, "expected_value_str", None)
    metadata = getattr(step, "metadata", None) or getattr(step, "metadata_json", None) or {}
    flow_info = metadata.get("flow_info") if isinstance(metadata, dict) else None

    return {
        "step_id": step.step_id,
        "step_index": getattr(step, "step_index", None),
        "step_name": step.step_name,
        "step_type": getattr(step, "step_type", "test"),
        "group_id": getattr(step, "group_id", None),
        "group_name": getattr(step, "group_name", None),
        "status": status_value,
        "measured_value": measured,
        "expected_value": expected,
        "lower_limit": getattr(step, "lower_limit", None),
        "upper_limit": getattr(step, "upper_limit", None),
        "passed": getattr(step, "passed", None),
        "duration_ms": getattr(step, "duration_ms", None),
        "error_code": getattr(step, "error_code", None),
        "error_message": getattr(step, "error_message", None),
        "retry_count": getattr(step, "retry_count", None),
        "loop_iteration": getattr(step, "loop_iteration", None),
        "loop_total": getattr(step, "loop_total", None),
        "adapter": getattr(step, "adapter", None),
        "comparison_operator": getattr(step, "comparison_operator", None),
        "flow_info": flow_info
    }
