"""
SeqMaster Runtime - Configuration
Layer: RUNTIME/CORE

Central configuration using Pydantic settings.
Supports environment variables and .env files.
"""

import os
from pathlib import Path
from typing import Optional
from pydantic_settings import BaseSettings
from pydantic import Field


class Settings(BaseSettings):
    """Application settings loaded from environment."""
    
    # Application
    APP_NAME: str = "SeqMaster Runtime"
    VERSION: str = "1.0.0"
    DEBUG: bool = False
    
    # API Server
    API_HOST: str = "0.0.0.0"
    API_PORT: int = 8000
    
    # Hostname for mDNS discovery
    HOSTNAME: str = Field(default_factory=lambda: os.uname().nodename if hasattr(os, 'uname') else "seqmaster-tester")
    TESTER_ID: str = Field(default_factory=lambda: os.environ.get("TESTER_ID", "TESTER-001"))
    
    # Database
    DATABASE_PATH: Path = Path.home() / ".seqmaster/data/seqmaster.db"
    DATABASE_URL: Optional[str] = None
    
    @property
    def database_url(self) -> str:
        if self.DATABASE_URL:
            return self.DATABASE_URL
        return f"sqlite+aiosqlite:///{self.DATABASE_PATH}"
    
    # File storage paths
    DATA_DIR: Path = Path.home() / ".seqmaster/data"
    SEQUENCES_DIR: Path = Path.home() / ".seqmaster/sequences"
    PROPERTY_SETS_DIR: Path = Path.home() / ".seqmaster/property_sets"
    LOGS_DIR: Path = Path.home() / ".seqmaster/logs"
    
    # Security
    SECRET_KEY: str = Field(default="CHANGE-ME-IN-PRODUCTION-32-CHARS")
    JWT_ALGORITHM: str = "HS256"
    ACCESS_TOKEN_EXPIRE_MINUTES: int = 60 * 24  # 24 hours
    
    # Watchdog
    WATCHDOG_TIMEOUT_SECONDS: int = 30
    POWER_LOSS_RECOVERY_ENABLED: bool = True
    
    # Hardware detection
    USB_POLL_INTERVAL_SECONDS: float = 5.0
    BLE_SCAN_TIMEOUT_SECONDS: float = 10.0
    
    # Execution
    DEFAULT_STEP_TIMEOUT_SECONDS: float = 30.0
    MAX_RETRY_COUNT: int = 3
    
    # mDNS / Zeroconf
    MDNS_SERVICE_TYPE: str = "_seqmaster._tcp.local."
    MDNS_SERVICE_NAME: str = Field(default_factory=lambda: f"SeqMaster-{os.environ.get('TESTER_ID', 'TESTER-001')}")
    
    # ========================================
    # Central Server (Option D Architecture)
    # ========================================
    
    # Central server URL (empty = sync disabled)
    CENTRAL_URL: Optional[str] = None  # e.g., "https://central.company.com"
    
    # API key for tester authentication with Central
    CENTRAL_API_KEY: Optional[str] = None
    
    # Site identifier for multi-site deployments
    SITE_ID: str = Field(default="default-site")
    
    # Sync settings
    SYNC_ENABLED: bool = True  # Enable/disable sync service
    SYNC_INTERVAL_SECONDS: int = 30  # How often to sync
    
    # ========================================
    # Licensing
    # ========================================
    
    # URL til licenskøb (hardware_id tilføjes automatisk)
    # Opdater når domæne er købt
    LICENSE_PURCHASE_URL: str = "http://91.98.205.173:3000/buy"
    
    class Config:
        env_file = ".env"
        env_prefix = "VERIMATRIX_"
        case_sensitive = True


# Global settings instance
settings = Settings()

# Ensure directories exist
def ensure_directories():
    """Create required directories if they don't exist."""
    for dir_path in [
        settings.DATA_DIR,
        settings.SEQUENCES_DIR,
        settings.PROPERTY_SETS_DIR,
        settings.LOGS_DIR
    ]:
        dir_path.mkdir(parents=True, exist_ok=True)
