"""
SeqMaster Runtime - mDNS Discovery Service
Layer: RUNTIME/CORE

Provides network discovery via mDNS/Zeroconf so the interface
can find RPi5 testers on the local network.
"""

import asyncio
import socket
from typing import Optional

import structlog
from zeroconf import ServiceInfo, Zeroconf
from zeroconf.asyncio import AsyncZeroconf

from src.core.config import settings

logger = structlog.get_logger(__name__)


class DiscoveryService:
    """
    mDNS/Zeroconf service for network discovery.
    
    Registers the tester on the local network so the interface
    application can discover it automatically.
    """
    
    def __init__(self):
        self._zeroconf: Optional[AsyncZeroconf] = None
        self._service_info: Optional[ServiceInfo] = None
    
    def _get_local_ip(self) -> str:
        """Get the local IP address."""
        try:
            s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            s.connect(("8.8.8.8", 80))
            ip = s.getsockname()[0]
            s.close()
            return ip
        except Exception:
            return "127.0.0.1"
    
    async def register(self) -> None:
        """Register the service with mDNS."""
        try:
            local_ip = self._get_local_ip()
            
            # Build service properties
            properties = {
                b"version": settings.VERSION.encode(),
                b"tester_id": settings.TESTER_ID.encode(),
                b"hostname": settings.HOSTNAME.encode(),
                b"api_port": str(settings.API_PORT).encode(),
            }
            
            self._service_info = ServiceInfo(
                type_=settings.MDNS_SERVICE_TYPE,
                name=f"{settings.MDNS_SERVICE_NAME}.{settings.MDNS_SERVICE_TYPE}",
                addresses=[socket.inet_aton(local_ip)],
                port=settings.API_PORT,
                properties=properties,
                server=f"{settings.HOSTNAME}.local."
            )
            
            self._zeroconf = AsyncZeroconf()
            await self._zeroconf.async_register_service(self._service_info)
            
            logger.info(
                "mDNS service registered",
                service_name=settings.MDNS_SERVICE_NAME,
                ip=local_ip,
                port=settings.API_PORT
            )
            
        except Exception as e:
            logger.error("Failed to register mDNS service", error=str(e))
    
    async def unregister(self) -> None:
        """Unregister the service from mDNS."""
        try:
            if self._zeroconf and self._service_info:
                await self._zeroconf.async_unregister_service(self._service_info)
                await self._zeroconf.async_close()
                logger.info("mDNS service unregistered")
        except Exception as e:
            logger.error("Failed to unregister mDNS service", error=str(e))


class TesterDiscoveryClient:
    """
    Client for discovering SeqMaster testers on the network.
    Used by the interface application.
    """
    
    def __init__(self):
        self._zeroconf: Optional[Zeroconf] = None
        self._discovered_testers: dict = {}
    
    async def scan(self, timeout: float = 5.0) -> list:
        """
        Scan for SeqMaster testers on the network.
        
        Args:
            timeout: Scan timeout in seconds
            
        Returns:
            List of discovered testers with their properties
        """
        from zeroconf import ServiceBrowser, ServiceListener
        
        testers = []
        
        class Listener(ServiceListener):
            def add_service(self, zc, type_, name):
                info = zc.get_service_info(type_, name)
                if info:
                    testers.append({
                        "name": name,
                        "addresses": [socket.inet_ntoa(addr) for addr in info.addresses],
                        "port": info.port,
                        "properties": {
                            k.decode(): v.decode() if isinstance(v, bytes) else v
                            for k, v in info.properties.items()
                        }
                    })
            
            def remove_service(self, zc, type_, name):
                pass
            
            def update_service(self, zc, type_, name):
                pass
        
        zc = Zeroconf()
        listener = Listener()
        browser = ServiceBrowser(zc, settings.MDNS_SERVICE_TYPE, listener)
        
        await asyncio.sleep(timeout)
        
        zc.close()
        
        return testers
