"""
Hardware Fingerprint Module

Genererer en unik identifikator for denne Raspberry Pi baseret på:
- CPU Serial Number (unik per Pi, kan ikke ændres)
- Primary MAC Address (backup identifikation)

Fingerprint bruges til at binde licensfiler til specifik hardware.
"""

import re
from pathlib import Path
from typing import Optional
import logging

logger = logging.getLogger(__name__)


def get_cpu_serial() -> Optional[str]:
    """
    Læser CPU serial number fra /proc/cpuinfo.
    
    Returns:
        16-tegn hex string (f.eks. "10000000abcd1234") eller None hvis ikke fundet.
    
    Eksempel fra /proc/cpuinfo:
        Serial          : 10000000abcd1234
    """
    try:
        cpuinfo = Path("/proc/cpuinfo").read_text()
        match = re.search(r"Serial\s*:\s*([0-9a-fA-F]+)", cpuinfo)
        if match:
            return match.group(1).lower()
    except Exception as e:
        logger.warning(f"Kunne ikke læse CPU serial: {e}")
    return None


def get_primary_mac() -> Optional[str]:
    """
    Læser MAC-adresse fra primary network interface.
    
    Returns:
        MAC-adresse uden koloner (f.eks. "dca632123456") eller None.
    """
    # Prøv eth0 først, derefter wlan0, end0 (RPi5 bruger end0)
    for interface in ["eth0", "wlan0", "end0"]:
        try:
            mac_path = Path(f"/sys/class/net/{interface}/address")
            if mac_path.exists():
                mac = mac_path.read_text().strip().replace(":", "").lower()
                if mac and mac != "000000000000":
                    return mac
        except Exception as e:
            logger.debug(f"Kunne ikke læse MAC fra {interface}: {e}")
    return None


def get_hardware_id() -> str:
    """
    Genererer det primære hardware ID brugt til licensering.
    
    Prioritet:
    1. CPU Serial (foretrukken - unik og permanent)
    2. MAC Address (fallback)
    3. "UNKNOWN" (fejltilstand)
    
    Returns:
        Hardware ID string brugt i licensfiler.
    """
    cpu_serial = get_cpu_serial()
    if cpu_serial:
        return cpu_serial
    
    mac = get_primary_mac()
    if mac:
        logger.warning("CPU serial ikke tilgængelig, bruger MAC adresse")
        return f"MAC-{mac}"
    
    logger.error("Kunne ikke bestemme hardware ID!")
    return "UNKNOWN"


def get_hardware_fingerprint() -> dict:
    """
    Returnerer komplet hardware fingerprint til diagnostik.
    
    Returns:
        Dict med alle tilgængelige hardware identifikatorer.
    """
    return {
        "hardware_id": get_hardware_id(),
        "cpu_serial": get_cpu_serial(),
        "mac_address": get_primary_mac(),
    }
