"""
License Service

Håndterer validering og håndhævelse af SeqMaster licenser.

Nøglebegreber:
- Licensfiler er JSON signeret med Ed25519
- Hardware ID binder licens til specifik enhed
- Demo mode aktiveres ved manglende/ugyldig licens

Eksempel brug:
    from src.core.license import get_license_service, LicenseTier
    
    service = get_license_service()
    info = service.get_license_info()
    
    if info.tier == LicenseTier.DEMO:
        show_demo_banner()
    
    if service.has_feature("analytics"):
        enable_analytics()
"""

import json
import base64
from datetime import datetime, timezone
from pathlib import Path
from typing import Optional
from dataclasses import dataclass, field
from enum import Enum
import logging

from cryptography.hazmat.primitives.asymmetric.ed25519 import Ed25519PublicKey
from cryptography.hazmat.primitives.serialization import load_pem_public_key
from cryptography.exceptions import InvalidSignature

from .fingerprint import get_hardware_id
from .config import settings

logger = logging.getLogger(__name__)


class LicenseTier(str, Enum):
    """Licensniveauer med stigende funktionalitet."""
    DEMO = "demo"
    BASIC = "basic"
    PROFESSIONAL = "professional"
    ENTERPRISE = "enterprise"


@dataclass
class LicenseInfo:
    """Parsed og valideret licensinformation."""
    tier: LicenseTier = LicenseTier.DEMO
    license_id: Optional[str] = None
    customer_name: Optional[str] = None
    customer_email: Optional[str] = None
    hardware_id: Optional[str] = None
    features: list[str] = field(default_factory=list)
    issued_at: Optional[datetime] = None
    expires_at: Optional[datetime] = None
    is_valid: bool = False
    error: Optional[str] = None
    
    def to_dict(self) -> dict:
        """Konverter til dictionary for API response."""
        return {
            "tier": self.tier.value,
            "license_id": self.license_id,
            "customer_name": self.customer_name,
            "customer_email": self.customer_email,
            "hardware_id": self.hardware_id,
            "features": self.features,
            "issued_at": self.issued_at.isoformat() if self.issued_at else None,
            "expires_at": self.expires_at.isoformat() if self.expires_at else None,
            "is_valid": self.is_valid,
            "error": self.error,
            "days_remaining": self._days_remaining(),
        }
    
    def _days_remaining(self) -> Optional[int]:
        """Beregn dage til udløb."""
        if not self.expires_at:
            return None
        delta = self.expires_at - datetime.now(timezone.utc)
        return max(0, delta.days)


# ============================================================================
# PUBLIC KEY - ERSTAT MED DIN EGEN!
# ============================================================================
# Generer nøglepar med kommandoen i LICENSING.md
# Private key gemmes sikkert på din server
# Public key indsættes her
# ============================================================================

PUBLIC_KEY_PEM = """-----BEGIN PUBLIC KEY-----
MCowBQYDK2VwAyEA/SESA+NgM0NBswKZ3Pstf0QQ504ldDTH4K1XLA7z22s=
-----END PUBLIC KEY-----
"""

# Sæt til True når du har indsat din rigtige public key
_PUBLIC_KEY_CONFIGURED = True


class LicenseService:
    """
    Service til håndtering af SeqMaster licenser.
    
    Brug:
        service = LicenseService()
        info = service.get_license_info()
        
        if info.tier == LicenseTier.DEMO:
            show_demo_banner()
        
        if service.has_feature("analytics"):
            enable_analytics()
    """
    
    def __init__(self, license_path: Optional[Path] = None):
        """
        Initialiser LicenseService.
        
        Args:
            license_path: Sti til licensfil. Default: runtime/data/license.sig
        """
        if license_path:
            self.license_path = license_path
        else:
            self.license_path = Path(settings.DATA_DIR) / "license.sig"
        
        self._cached_info: Optional[LicenseInfo] = None
        self._public_key = self._load_public_key()
    
    def _load_public_key(self) -> Optional[Ed25519PublicKey]:
        """Indlæs public key til signaturverifikation."""
        if not _PUBLIC_KEY_CONFIGURED:
            logger.warning("Public key ikke konfigureret - licensvalidering deaktiveret")
            return None
        
        try:
            key = load_pem_public_key(PUBLIC_KEY_PEM.encode())
            if isinstance(key, Ed25519PublicKey):
                return key
            else:
                logger.error("Public key er ikke Ed25519 format")
        except Exception as e:
            logger.error(f"Kunne ikke indlæse public key: {e}")
        return None
    
    def get_license_info(self, force_reload: bool = False) -> LicenseInfo:
        """
        Hent aktuel licensinformation.
        
        Args:
            force_reload: Genindlæs fra fil selvom cached version findes.
        
        Returns:
            LicenseInfo med aktuel status.
        """
        if self._cached_info and not force_reload:
            return self._cached_info
        
        self._cached_info = self._validate_license()
        return self._cached_info
    
    def _validate_license(self) -> LicenseInfo:
        """Valider licensfil og returner info."""
        
        # Public key ikke konfigureret = development mode (fuld adgang)
        if not _PUBLIC_KEY_CONFIGURED:
            logger.info("Development mode - fuld licens aktiveret")
            return LicenseInfo(
                tier=LicenseTier.ENTERPRISE,
                license_id="DEV-LICENSE",
                customer_name="Development Mode",
                hardware_id=get_hardware_id(),
                features=list(FEATURE_DESCRIPTIONS.keys()),
                is_valid=True,
                error=None
            )
        
        # Ingen licensfil = demo
        if not self.license_path.exists():
            return LicenseInfo(
                tier=LicenseTier.DEMO,
                hardware_id=get_hardware_id(),
                error="Ingen licensfil fundet"
            )
        
        try:
            # Læs licensfil
            license_data = json.loads(self.license_path.read_text())
            license_content = license_data.get("license", {})
            signature_b64 = license_data.get("signature", "")
            
            # Verificer signatur
            if not self._verify_signature(license_content, signature_b64):
                return LicenseInfo(
                    tier=LicenseTier.DEMO,
                    hardware_id=get_hardware_id(),
                    error="Ugyldig signatur"
                )
            
            # Tjek hardware ID
            expected_hw_id = license_content.get("hardware_id", "")
            actual_hw_id = get_hardware_id()
            
            if expected_hw_id.lower() != actual_hw_id.lower():
                return LicenseInfo(
                    tier=LicenseTier.DEMO,
                    hardware_id=actual_hw_id,
                    error=f"Hardware ID matcher ikke (forventet: {expected_hw_id}, faktisk: {actual_hw_id})"
                )
            
            # Tjek udløbsdato
            expires_str = license_content.get("expires_at")
            expires_at = None
            if expires_str:
                expires_at = datetime.fromisoformat(expires_str.replace("Z", "+00:00"))
                if expires_at < datetime.now(timezone.utc):
                    return LicenseInfo(
                        tier=LicenseTier.DEMO,
                        license_id=license_content.get("id"),
                        hardware_id=actual_hw_id,
                        expires_at=expires_at,
                        error="Licens er udløbet"
                    )
            
            # Parse issued_at
            issued_str = license_content.get("issued_at")
            issued_at = None
            if issued_str:
                issued_at = datetime.fromisoformat(issued_str.replace("Z", "+00:00"))
            
            # Alt OK - returner fuld info
            customer = license_content.get("customer", {})
            tier_str = license_content.get("tier", "basic")
            
            return LicenseInfo(
                tier=LicenseTier(tier_str),
                license_id=license_content.get("id"),
                customer_name=customer.get("name"),
                customer_email=customer.get("email"),
                hardware_id=expected_hw_id,
                features=license_content.get("features", []),
                issued_at=issued_at,
                expires_at=expires_at,
                is_valid=True,
                error=None
            )
            
        except json.JSONDecodeError as e:
            return LicenseInfo(
                tier=LicenseTier.DEMO,
                hardware_id=get_hardware_id(),
                error=f"Ugyldig licensfil format: {e}"
            )
        except Exception as e:
            logger.exception("Fejl ved licensvalidering")
            return LicenseInfo(
                tier=LicenseTier.DEMO,
                hardware_id=get_hardware_id(),
                error=f"Valideringsfejl: {e}"
            )
    
    def _verify_signature(self, content: dict, signature_b64: str) -> bool:
        """Verificer Ed25519 signatur."""
        if not self._public_key:
            logger.error("Public key ikke tilgængelig")
            return False
        
        try:
            # Signatur er over JSON-encoded content (deterministisk)
            content_bytes = json.dumps(
                content, 
                sort_keys=True, 
                separators=(',', ':'),
                ensure_ascii=False
            ).encode('utf-8')
            signature = base64.b64decode(signature_b64)
            
            self._public_key.verify(signature, content_bytes)
            return True
            
        except InvalidSignature:
            logger.warning("Signaturverifikation fejlede")
            return False
        except Exception as e:
            logger.error(f"Signaturfejl: {e}")
            return False
    
    def has_feature(self, feature: str) -> bool:
        """
        Tjek om en specifik feature er aktiveret.
        
        Args:
            feature: Feature navn (f.eks. "analytics", "central_integration")
        
        Returns:
            True hvis feature er inkluderet i licensen.
        """
        info = self.get_license_info()
        
        # Demo har basis features
        if info.tier == LicenseTier.DEMO:
            return feature in DEMO_FEATURES
        
        return feature in info.features
    
    def is_demo(self) -> bool:
        """Returner True hvis systemet kører i demo mode."""
        return self.get_license_info().tier == LicenseTier.DEMO
    
    def get_demo_limits(self) -> dict:
        """Hent demo mode begrænsninger."""
        return DEMO_LIMITS.copy()
    
    def check_demo_limit(self, limit_name: str, current_value: int) -> bool:
        """
        Tjek om en demo-grænse er overskredet.
        
        Args:
            limit_name: Navn på begrænsning (f.eks. "max_steps")
            current_value: Aktuel værdi
        
        Returns:
            True hvis inden for grænsen (eller ikke i demo mode)
        """
        if not self.is_demo():
            return True
        
        limit = DEMO_LIMITS.get(limit_name)
        if limit is None:
            return True
        
        return current_value <= limit
    
    def upload_license(self, license_content: bytes) -> LicenseInfo:
        """
        Upload og valider ny licensfil.
        
        Args:
            license_content: Rå bytes fra uploadet fil.
        
        Returns:
            LicenseInfo med status efter upload.
        
        Raises:
            ValueError: Hvis filen er ugyldig.
        """
        # Gem til temp fil først
        temp_path = self.license_path.with_suffix(".tmp")
        temp_path.write_bytes(license_content)
        
        try:
            # Valider med temp fil
            temp_service = LicenseService(license_path=temp_path)
            temp_service._cached_info = None  # Force reload
            info = temp_service._validate_license()
            
            if info.is_valid:
                # Flyt til rigtig placering
                temp_path.rename(self.license_path)
                self._cached_info = None  # Clear cache
                logger.info(f"Licens aktiveret: {info.license_id} ({info.tier.value})")
                return self.get_license_info(force_reload=True)
            else:
                raise ValueError(info.error or "Ugyldig licensfil")
                
        finally:
            # Slet temp fil hvis den stadig findes
            temp_path.unlink(missing_ok=True)
    
    def remove_license(self) -> bool:
        """
        Fjern aktuel licensfil (går tilbage til demo mode).
        
        Returns:
            True hvis licens blev fjernet.
        """
        if self.license_path.exists():
            self.license_path.unlink()
            self._cached_info = None
            logger.info("Licens fjernet - demo mode aktiveret")
            return True
        return False


# ============================================================================
# DEMO MODE KONFIGURATION
# ============================================================================

# Features tilgængelige i demo mode
DEMO_FEATURES = [
    "basic_sequences",      # Kan køre sekvenser
    "manual_results",       # Kan se resultater manuelt
]

# Demo mode begrænsninger
DEMO_LIMITS = {
    "max_steps": 10,           # Max steps per sekvens
    "max_results": 10,         # Max gemte testresultater
    "max_sequences": 3,        # Max antal sekvenser
}

# Feature beskrivelser til UI
FEATURE_DESCRIPTIONS = {
    "basic_sequences": "Kør test sekvenser",
    "unlimited_sequences": "Ubegrænset antal sekvenser",
    "unlimited_steps": "Ubegrænset steps per sekvens",
    "unlimited_results": "Gem alle testresultater",
    "analytics": "Statistik og rapporter",
    "central_integration": "Central server integration",
    "priority_support": "Prioriteret support",
    "custom_drivers": "Custom hardware drivers",
    "api_access": "Ekstern API adgang",
    "multi_station": "Multi-station koordinering",
}

# Tier feature mappings
TIER_FEATURES = {
    LicenseTier.DEMO: DEMO_FEATURES,
    LicenseTier.BASIC: [
        "basic_sequences",
        "manual_results",
        "unlimited_results",
    ],
    LicenseTier.PROFESSIONAL: [
        "basic_sequences",
        "manual_results",
        "unlimited_sequences",
        "unlimited_steps",
        "unlimited_results",
        "analytics",
        "central_integration",
        "api_access",
        "priority_support",
    ],
    LicenseTier.ENTERPRISE: list(FEATURE_DESCRIPTIONS.keys()),
}


# ============================================================================
# GLOBAL SINGLETON
# ============================================================================

_license_service: Optional[LicenseService] = None


def get_license_service() -> LicenseService:
    """Hent global LicenseService instance."""
    global _license_service
    if _license_service is None:
        _license_service = LicenseService()
    return _license_service


def reset_license_service() -> None:
    """Reset global LicenseService (til testing)."""
    global _license_service
    _license_service = None
