"""
SeqMaster Runtime - Service Registry
Layer: RUNTIME/CORE

Simple service registry for dependency injection and service access.
"""

from typing import Any, Dict, Optional


class ServiceRegistry:
    """
    Global service registry for managing singleton services.
    Provides a simple way to access services across the application.
    """
    
    _services: Dict[str, Any] = {}
    
    @classmethod
    def set(cls, name: str, service: Any) -> None:
        """Register a service."""
        cls._services[name] = service
    
    @classmethod
    def get(cls, name: str) -> Optional[Any]:
        """Get a registered service."""
        return cls._services.get(name)
    
    @classmethod
    def require(cls, name: str) -> Any:
        """Get a registered service or raise error if not found."""
        service = cls._services.get(name)
        if service is None:
            raise RuntimeError(f"Service '{name}' not registered")
        return service
    
    @classmethod
    def clear(cls) -> None:
        """Clear all registered services."""
        cls._services.clear()
