"""
SeqMaster Runtime - Tester Lock System
Layer: Core

Provides mutex/semaphore functionality to ensure only one operator
can control the tester at a time. Prevents race conditions when
multiple browser tabs or clients try to run tests simultaneously.

Usage:
    from src.core.tester_lock import TesterLock
    
    lock = TesterLock()
    if lock.acquire("client-123", "operator-token"):
        # Has exclusive access
        ...
        lock.release("client-123")
"""

import asyncio
from datetime import datetime, timedelta
from typing import Optional, Dict, Any
from dataclasses import dataclass, field
import logging

logger = logging.getLogger(__name__)


@dataclass
class LockInfo:
    """Information about the current lock holder."""
    client_id: str
    operator_id: Optional[str] = None
    operator_name: Optional[str] = None
    session_id: Optional[str] = None
    acquired_at: datetime = field(default_factory=datetime.utcnow)
    last_heartbeat: datetime = field(default_factory=datetime.utcnow)
    test_running: bool = False
    

class TesterLock:
    """
    Singleton mutex for tester reservation.
    
    Ensures only one client can control the tester at a time.
    Features:
    - Heartbeat-based timeout (auto-release after inactivity)
    - Session tracking for reconnection
    - WebSocket broadcast on lock state changes
    """
    
    _instance: Optional['TesterLock'] = None
    
    # Timeout settings
    LOCK_TIMEOUT_SECONDS = 120  # 2 minutes without heartbeat
    HEARTBEAT_INTERVAL = 30     # Expected heartbeat every 30 seconds
    WARNING_THRESHOLD = 90      # Warn at 1.5 minutes
    
    def __new__(cls):
        if cls._instance is None:
            cls._instance = super().__new__(cls)
            cls._instance._initialized = False
        return cls._instance
    
    def __init__(self):
        if self._initialized:
            return
        self._initialized = True
        
        self._lock: Optional[LockInfo] = None
        self._lock_mutex = asyncio.Lock()
        self._broadcast_callback = None
        self._pending_release: Optional[asyncio.Task] = None
        
        logger.info("TesterLock initialized")
    
    def set_broadcast_callback(self, callback):
        """Set callback for broadcasting lock state changes via WebSocket."""
        self._broadcast_callback = callback
    
    async def _broadcast_state(self, event: str, data: Dict[str, Any] = None):
        """Broadcast lock state change to all connected clients."""
        if self._broadcast_callback:
            message = {
                "type": "tester_lock",
                "event": event,
                "lock_status": self.get_status(),
                **(data or {})
            }
            try:
                await self._broadcast_callback(message)
            except Exception as e:
                logger.error(f"Failed to broadcast lock state: {e}")
    
    async def acquire(
        self, 
        client_id: str, 
        operator_id: Optional[str] = None,
        operator_name: Optional[str] = None,
        force: bool = False
    ) -> Dict[str, Any]:
        """
        Attempt to acquire the tester lock.
        
        Args:
            client_id: Unique identifier for the client (browser tab)
            operator_id: Optional user ID of the operator
            operator_name: Optional display name of the operator
            force: If True, forcefully take the lock (admin only)
            
        Returns:
            Dict with 'acquired' bool and 'lock_info' or 'error'
        """
        async with self._lock_mutex:
            # Check if already locked
            if self._lock is not None:
                # Same client requesting again - refresh heartbeat
                if self._lock.client_id == client_id:
                    self._lock.last_heartbeat = datetime.utcnow()
                    logger.debug(f"Lock refreshed for client {client_id}")
                    return {
                        "acquired": True,
                        "refreshed": True,
                        "lock_info": self._get_lock_dict()
                    }
                
                # Check if existing lock has timed out
                if self._is_timed_out():
                    logger.info(f"Lock timed out for {self._lock.client_id}, releasing")
                    old_holder = self._lock.client_id
                    self._lock = None
                    await self._broadcast_state("timeout", {"previous_holder": old_holder})
                elif not force:
                    # Lock is held by another client
                    logger.warning(f"Lock denied for {client_id}, held by {self._lock.client_id}")
                    return {
                        "acquired": False,
                        "error": "tester_busy",
                        "message": "Tester is currently in use by another operator",
                        "holder": {
                            "operator_name": self._lock.operator_name or "Unknown",
                            "acquired_at": self._lock.acquired_at.isoformat(),
                            "test_running": self._lock.test_running
                        }
                    }
                else:
                    # Force acquire - release existing lock first
                    logger.warning(f"Force acquiring lock from {self._lock.client_id} for {client_id}")
                    old_holder = self._lock.client_id
                    await self._broadcast_state("force_released", {"previous_holder": old_holder})
            
            # Acquire the lock
            self._lock = LockInfo(
                client_id=client_id,
                operator_id=operator_id,
                operator_name=operator_name
            )
            
            # Cancel any pending auto-release
            if self._pending_release:
                self._pending_release.cancel()
                self._pending_release = None
            
            # Start timeout checker
            self._pending_release = asyncio.create_task(self._check_timeout())
            
            logger.info(f"Lock acquired by {client_id} ({operator_name or 'anonymous'})")
            await self._broadcast_state("acquired")
            
            return {
                "acquired": True,
                "lock_info": self._get_lock_dict()
            }
    
    async def release(self, client_id: str, force: bool = False) -> Dict[str, Any]:
        """
        Release the tester lock.
        
        Args:
            client_id: The client requesting release
            force: If True, release regardless of holder (admin only)
            
        Returns:
            Dict with 'released' bool
        """
        async with self._lock_mutex:
            if self._lock is None:
                return {"released": True, "message": "No lock was held"}
            
            if self._lock.client_id != client_id and not force:
                logger.warning(f"Release denied: {client_id} is not the lock holder")
                return {
                    "released": False,
                    "error": "not_lock_holder",
                    "message": "You are not the current lock holder"
                }
            
            old_holder = self._lock.client_id
            self._lock = None
            
            # Cancel timeout checker
            if self._pending_release:
                self._pending_release.cancel()
                self._pending_release = None
            
            logger.info(f"Lock released by {client_id}")
            await self._broadcast_state("released", {"previous_holder": old_holder})
            
            return {"released": True}
    
    async def heartbeat(self, client_id: str) -> Dict[str, Any]:
        """
        Send heartbeat to keep lock alive.
        
        Args:
            client_id: The client sending heartbeat
            
        Returns:
            Dict with status and remaining time
        """
        async with self._lock_mutex:
            if self._lock is None:
                return {
                    "success": False,
                    "error": "no_lock",
                    "message": "No lock is currently held"
                }
            
            if self._lock.client_id != client_id:
                return {
                    "success": False,
                    "error": "not_lock_holder",
                    "message": "You are not the current lock holder"
                }
            
            self._lock.last_heartbeat = datetime.utcnow()
            remaining = self.LOCK_TIMEOUT_SECONDS
            
            logger.debug(f"Heartbeat received from {client_id}")
            
            return {
                "success": True,
                "remaining_seconds": remaining,
                "lock_info": self._get_lock_dict()
            }
    
    async def set_test_running(self, client_id: str, running: bool, session_id: str = None):
        """Update lock to indicate test is running."""
        async with self._lock_mutex:
            if self._lock and self._lock.client_id == client_id:
                self._lock.test_running = running
                if session_id:
                    self._lock.session_id = session_id
                await self._broadcast_state("test_state_changed")
    
    def get_status(self) -> Dict[str, Any]:
        """Get current lock status."""
        if self._lock is None:
            return {
                "locked": False,
                "available": True
            }
        
        time_held = (datetime.utcnow() - self._lock.acquired_at).total_seconds()
        time_since_heartbeat = (datetime.utcnow() - self._lock.last_heartbeat).total_seconds()
        
        return {
            "locked": True,
            "available": False,
            "holder": {
                "client_id": self._lock.client_id,
                "operator_id": self._lock.operator_id,
                "operator_name": self._lock.operator_name or "Unknown",
                "session_id": self._lock.session_id,
                "test_running": self._lock.test_running
            },
            "acquired_at": self._lock.acquired_at.isoformat(),
            "time_held_seconds": int(time_held),
            "last_heartbeat": self._lock.last_heartbeat.isoformat(),
            "seconds_since_heartbeat": int(time_since_heartbeat),
            "timeout_in_seconds": max(0, self.LOCK_TIMEOUT_SECONDS - int(time_since_heartbeat))
        }
    
    def is_locked(self) -> bool:
        """Check if tester is currently locked."""
        if self._lock is None:
            return False
        if self._is_timed_out():
            return False
        return True
    
    def is_holder(self, client_id: str) -> bool:
        """Check if given client holds the lock."""
        return self._lock is not None and self._lock.client_id == client_id
    
    def _is_timed_out(self) -> bool:
        """Check if current lock has timed out."""
        if self._lock is None:
            return False
        elapsed = (datetime.utcnow() - self._lock.last_heartbeat).total_seconds()
        return elapsed > self.LOCK_TIMEOUT_SECONDS
    
    def _get_lock_dict(self) -> Dict[str, Any]:
        """Get lock info as dictionary."""
        if self._lock is None:
            return None
        return {
            "client_id": self._lock.client_id,
            "operator_id": self._lock.operator_id,
            "operator_name": self._lock.operator_name,
            "session_id": self._lock.session_id,
            "acquired_at": self._lock.acquired_at.isoformat(),
            "test_running": self._lock.test_running
        }
    
    async def _check_timeout(self):
        """Background task to check for lock timeout."""
        try:
            while True:
                await asyncio.sleep(30)  # Check every 30 seconds
                
                async with self._lock_mutex:
                    if self._lock is None:
                        break
                    
                    elapsed = (datetime.utcnow() - self._lock.last_heartbeat).total_seconds()
                    
                    # Warn approaching timeout
                    if elapsed > self.WARNING_THRESHOLD and elapsed < self.LOCK_TIMEOUT_SECONDS:
                        remaining = self.LOCK_TIMEOUT_SECONDS - elapsed
                        logger.warning(f"Lock timeout warning: {remaining:.0f}s remaining")
                        await self._broadcast_state("timeout_warning", {
                            "remaining_seconds": int(remaining)
                        })
                    
                    # Timeout - release lock
                    elif elapsed >= self.LOCK_TIMEOUT_SECONDS:
                        logger.info(f"Lock timed out for {self._lock.client_id}")
                        old_holder = self._lock.client_id
                        self._lock = None
                        await self._broadcast_state("timeout", {"previous_holder": old_holder})
                        break
                        
        except asyncio.CancelledError:
            pass
        except Exception as e:
            logger.error(f"Error in timeout checker: {e}")


# Global instance
_tester_lock: Optional[TesterLock] = None


def get_tester_lock() -> TesterLock:
    """Get the global TesterLock instance."""
    global _tester_lock
    if _tester_lock is None:
        _tester_lock = TesterLock()
    return _tester_lock
