"""
SeqMaster Runtime - Database Connection
Layer: DATABASE

Async database connection management using SQLAlchemy.
"""

from contextlib import asynccontextmanager
from typing import AsyncGenerator

from sqlalchemy.ext.asyncio import AsyncSession, create_async_engine, async_sessionmaker
from sqlalchemy.pool import StaticPool

from src.core.config import settings
from src.database.models import Base


def _apply_sqlite_migrations(conn) -> None:
    """Apply lightweight SQLite migrations (columns + indexes)."""
    # Add missing columns for step_results
    result = conn.exec_driver_sql("PRAGMA table_info('step_results')")
    existing_columns = {row[1] for row in result.fetchall()}

    if "adapter" not in existing_columns:
        conn.exec_driver_sql("ALTER TABLE step_results ADD COLUMN adapter VARCHAR(50)")
    if "comparison_operator" not in existing_columns:
        conn.exec_driver_sql("ALTER TABLE step_results ADD COLUMN comparison_operator VARCHAR(20)")

    # Create indexes to speed up analytics queries
    conn.exec_driver_sql(
        "CREATE INDEX IF NOT EXISTS idx_step_results_step_name_group "
        "ON step_results(step_name, group_name)"
    )
    conn.exec_driver_sql(
        "CREATE INDEX IF NOT EXISTS idx_step_results_completed_at "
        "ON step_results(completed_at)"
    )
    conn.exec_driver_sql(
        "CREATE INDEX IF NOT EXISTS idx_step_results_passed "
        "ON step_results(passed)"
    )
    conn.exec_driver_sql(
        "CREATE INDEX IF NOT EXISTS idx_step_results_step_type "
        "ON step_results(step_type)"
    )
    conn.exec_driver_sql(
        "CREATE INDEX IF NOT EXISTS idx_step_results_session_id "
        "ON step_results(session_id)"
    )

    # Test session indexes for filters
    conn.exec_driver_sql(
        "CREATE INDEX IF NOT EXISTS idx_test_sessions_started_at "
        "ON test_sessions(started_at)"
    )
    conn.exec_driver_sql(
        "CREATE INDEX IF NOT EXISTS idx_test_sessions_sequence_id "
        "ON test_sessions(sequence_id)"
    )
    conn.exec_driver_sql(
        "CREATE INDEX IF NOT EXISTS idx_test_sessions_dut_type "
        "ON test_sessions(dut_type)"
    )
    conn.exec_driver_sql(
        "CREATE INDEX IF NOT EXISTS idx_test_sessions_operator "
        "ON test_sessions(operator)"
    )

# Create async engine
engine = create_async_engine(
    settings.database_url,
    echo=settings.DEBUG,
    poolclass=StaticPool,  # Good for SQLite
    connect_args={"check_same_thread": False}  # Required for SQLite
)

# Session factory
async_session_maker = async_sessionmaker(
    engine,
    class_=AsyncSession,
    expire_on_commit=False
)


async def init_database() -> None:
    """Initialize database and create tables."""
    settings.DATA_DIR.mkdir(parents=True, exist_ok=True)
    
    async with engine.begin() as conn:
        await conn.run_sync(Base.metadata.create_all)
        await conn.run_sync(_apply_sqlite_migrations)


async def close_database() -> None:
    """Close database connections."""
    await engine.dispose()


@asynccontextmanager
async def get_session() -> AsyncGenerator[AsyncSession, None]:
    """Get database session context manager."""
    session = async_session_maker()
    try:
        yield session
        await session.commit()
    except Exception:
        await session.rollback()
        raise
    finally:
        await session.close()


async def get_db_session() -> AsyncGenerator[AsyncSession, None]:
    """Dependency for FastAPI endpoints."""
    async with get_session() as session:
        yield session


def get_engine():
    """Return the async engine instance."""
    return engine
