"""
SeqMaster Runtime - Database Models
Layer: DATABASE

SQLAlchemy models for local data storage.
"""

from datetime import datetime
from enum import Enum as PyEnum
from typing import Optional, List
import json

from sqlalchemy import (
    Column, Integer, String, Float, Boolean, DateTime, 
    Text, ForeignKey, Enum, JSON, LargeBinary
)
from sqlalchemy.orm import relationship, declarative_base

Base = declarative_base()


class UserRole(str, PyEnum):
    """User roles for access control."""
    OPERATOR = "operator"
    ENGINEER = "engineer"
    ADMIN = "admin"


class TestStatus(str, PyEnum):
    """Test execution status."""
    PENDING = "pending"
    RUNNING = "running"
    PAUSED = "paused"
    PASSED = "passed"
    FAILED = "failed"
    ERROR = "error"
    ABORTED = "aborted"
    SKIPPED = "skipped"


class StepStatus(str, PyEnum):
    """Individual step status."""
    PENDING = "pending"
    RUNNING = "running"
    PASSED = "passed"
    FAILED = "failed"
    ERROR = "error"
    SKIPPED = "skipped"
    RETRY = "retry"
    EXECUTED = "executed"  # Flow control step executed (neutral - not pass/fail)


# ============================================
# USER MANAGEMENT
# ============================================

class User(Base):
    """Local user database for authentication."""
    __tablename__ = "users"
    
    id = Column(Integer, primary_key=True, autoincrement=True)
    username = Column(String(100), unique=True, nullable=False, index=True)
    password_hash = Column(String(255), nullable=False)
    role = Column(Enum(UserRole), nullable=False, default=UserRole.OPERATOR)
    full_name = Column(String(200))
    email = Column(String(200))
    active = Column(Boolean, default=True)
    created_at = Column(DateTime, default=datetime.utcnow)
    updated_at = Column(DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    last_login = Column(DateTime)
    
    # Relationships
    test_sessions = relationship("TestSession", back_populates="operator_user")
    permissions = relationship("UserPermission", back_populates="user", cascade="all, delete-orphan")


class UserPermission(Base):
    """
    Custom permissions per user.
    Overrides default role-based permissions when set.
    """
    __tablename__ = "user_permissions"
    
    id = Column(Integer, primary_key=True, autoincrement=True)
    user_id = Column(Integer, ForeignKey("users.id"), nullable=False)
    permission = Column(String(100), nullable=False)  # e.g., "run_tests", "edit_sequences"
    granted = Column(Boolean, nullable=False)  # True = allowed, False = denied
    granted_by = Column(String(100))  # Who granted this permission
    granted_at = Column(DateTime, default=datetime.utcnow)
    
    # Relationships
    user = relationship("User", back_populates="permissions")
    
    # Unique constraint: one permission per user
    __table_args__ = (
        {'sqlite_autoincrement': True},
    )


class SystemConfig(Base):
    """
    System configuration stored in database.
    Used for setup state, preferences, etc.
    """
    __tablename__ = "system_config"
    
    id = Column(Integer, primary_key=True, autoincrement=True)
    key = Column(String(100), unique=True, nullable=False)
    value = Column(Text)
    updated_at = Column(DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)


# ============================================
# SEQUENCE AND PROPERTY SET VERSIONING
# ============================================

class Sequence(Base):
    """Test sequence version record."""
    __tablename__ = "sequences"
    
    id = Column(Integer, primary_key=True, autoincrement=True)
    sequence_id = Column(String(100), nullable=False, index=True)
    version = Column(String(50), nullable=False)
    name = Column(String(200), nullable=False)
    description = Column(Text)
    
    # Version control
    status = Column(String(20), default="draft")  # draft or published
    
    # DUT metadata
    dut_description = Column(Text)
    dut_hw_version = Column(String(100))
    dut_partnumber = Column(String(100))
    dut_batch_number = Column(String(100))
    
    content_hash = Column(String(64))  # SHA-256 of content
    content = Column(Text, nullable=False)  # JSON/YAML content
    created_at = Column(DateTime, default=datetime.utcnow)
    published_at = Column(DateTime)  # When published
    created_by = Column(String(100))
    active = Column(Boolean, default=True)
    
    # Relationships
    test_sessions = relationship("TestSession", back_populates="sequence")
    error_codes = relationship("StepErrorCode", back_populates="sequence", cascade="all, delete-orphan")


class StepErrorCode(Base):
    """
    Error code definitions for test steps.
    
    Each step can have multiple error codes, each applicable to a specific
    measurement range. Used to provide operators with troubleshooting info.
    """
    __tablename__ = "step_error_codes"
    
    id = Column(Integer, primary_key=True, autoincrement=True)
    sequence_id = Column(Integer, ForeignKey("sequences.id"), nullable=False)
    
    # Step identification
    step_id = Column(String(50), nullable=False, index=True)  # e.g., STEP001
    error_code_id = Column(String(50), nullable=False)  # e.g., ERR1234ABC
    
    # Error details
    title = Column(String(200), nullable=False)
    description = Column(Text)
    image = Column(Text)  # Base64 encoded image or URL
    
    # Measurement range for this error (optional)
    range_min = Column(Float)  # Show this error when measured value >= range_min
    range_max = Column(Float)  # Show this error when measured value <= range_max
    
    # Ordering
    display_order = Column(Integer, default=0)
    
    # Timestamps
    created_at = Column(DateTime, default=datetime.utcnow)
    updated_at = Column(DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    
    # Relationships
    sequence = relationship("Sequence", back_populates="error_codes")


class PropertySet(Base):
    """Property set (limits) version record."""
    __tablename__ = "property_sets"
    
    id = Column(Integer, primary_key=True, autoincrement=True)
    property_set_id = Column(String(100), nullable=False, index=True)
    version = Column(String(50), nullable=False)
    name = Column(String(200), nullable=False)
    dut_type = Column(String(100))
    dut_revision = Column(String(50))
    content_hash = Column(String(64))
    content = Column(Text, nullable=False)  # JSON content
    created_at = Column(DateTime, default=datetime.utcnow)
    created_by = Column(String(100))
    active = Column(Boolean, default=True)
    
    # Relationships
    test_sessions = relationship("TestSession", back_populates="property_set")


# ============================================
# TEST EXECUTION RECORDS
# ============================================

class TestSession(Base):
    """Test session (one execution of a sequence)."""
    __tablename__ = "test_sessions"
    
    id = Column(Integer, primary_key=True, autoincrement=True)
    session_id = Column(String(100), unique=True, nullable=False, index=True)
    
    # DUT information
    dut_id = Column(String(100), nullable=False, index=True)
    dut_serial = Column(String(100), index=True)
    dut_type = Column(String(100))
    dut_revision = Column(String(50))
    
    # Sequence and property set references
    sequence_id = Column(Integer, ForeignKey("sequences.id"))
    sequence_version = Column(String(50))
    property_set_id = Column(Integer, ForeignKey("property_sets.id"))
    property_set_version = Column(String(50))
    
    # Execution info
    operator = Column(String(100))
    operator_id = Column(Integer, ForeignKey("users.id"))
    tester_id = Column(String(100))
    station_id = Column(String(100))
    
    # Timing
    started_at = Column(DateTime, default=datetime.utcnow)
    completed_at = Column(DateTime)
    duration_seconds = Column(Float)
    
    # Status
    status = Column(Enum(TestStatus), default=TestStatus.PENDING)
    current_step_index = Column(Integer, default=0)
    current_step_id = Column(String(50))
    progress_percent = Column(Float, default=0.0)
    
    # Results summary
    total_steps = Column(Integer, default=0)
    passed_steps = Column(Integer, default=0)
    failed_steps = Column(Integer, default=0)
    error_code = Column(String(50))
    error_message = Column(Text)
    
    # Metadata
    metadata_json = Column(JSON)
    
    # Relationships
    sequence = relationship("Sequence", back_populates="test_sessions")
    property_set = relationship("PropertySet", back_populates="test_sessions")
    operator_user = relationship("User", back_populates="test_sessions")
    step_results = relationship("StepResult", back_populates="session", 
                                cascade="all, delete-orphan")


class StepResult(Base):
    """Individual test step result."""
    __tablename__ = "step_results"
    
    id = Column(Integer, primary_key=True, autoincrement=True)
    session_id = Column(Integer, ForeignKey("test_sessions.id"), nullable=False)
    
    # Step identification
    step_id = Column(String(50), nullable=False)  # M### or TS###
    step_index = Column(Integer, nullable=False)
    step_name = Column(String(200))
    step_type = Column(String(50))  # test, action, flow
    group_id = Column(String(50))  # Parent TS### if this is a M### step
    group_name = Column(String(200))  # Name of parent group
    adapter = Column(String(50))  # python, exec, compare, driver
    comparison_operator = Column(String(20))  # eq, ne, gt, between, etc.
    
    # Timing
    started_at = Column(DateTime)
    completed_at = Column(DateTime)
    duration_ms = Column(Float)
    
    # Status
    status = Column(Enum(StepStatus), default=StepStatus.PENDING)
    retry_count = Column(Integer, default=0)
    
    # Measurement data
    measured_value = Column(Float)
    measured_value_str = Column(String(500))  # For non-numeric values
    expected_value = Column(Float)
    expected_value_str = Column(String(500))  # For string expected values
    lower_limit = Column(Float)
    upper_limit = Column(Float)
    unit = Column(String(50))
    
    # Result
    passed = Column(Boolean)
    error_code = Column(String(50))
    error_message = Column(Text)
    
    # Raw data
    raw_data = Column(LargeBinary)
    raw_data_type = Column(String(50))  # Format of raw data
    
    # Metadata
    metadata_json = Column(JSON)
    
    # Relationships
    session = relationship("TestSession", back_populates="step_results")


# ============================================
# AUDIT LOG
# ============================================

class AuditLog(Base):
    """Audit log for tracking all actions."""
    __tablename__ = "audit_logs"
    
    id = Column(Integer, primary_key=True, autoincrement=True)
    timestamp = Column(DateTime, default=datetime.utcnow, index=True)
    user_id = Column(Integer, ForeignKey("users.id"))
    username = Column(String(100))
    action = Column(String(100), nullable=False)
    resource_type = Column(String(100))  # sequence, property_set, test_session, etc.
    resource_id = Column(String(100))
    details = Column(JSON)
    ip_address = Column(String(50))


# ============================================
# SYSTEM STATE
# ============================================

class SystemState(Base):
    """Persistent system state for recovery."""
    __tablename__ = "system_state"
    
    id = Column(Integer, primary_key=True, autoincrement=True)
    key = Column(String(100), unique=True, nullable=False)
    value = Column(Text)
    updated_at = Column(DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)


# ============================================
# SYNC QUEUE (Option D: Offline-First Architecture)
# ============================================

class SyncStatus(str, PyEnum):
    """Sync queue item status."""
    PENDING = "pending"
    IN_PROGRESS = "in_progress"
    COMPLETED = "completed"
    FAILED = "failed"


class SyncQueue(Base):
    """
    Queue for pending uploads to Central server.
    
    Used for offline-first sync: data is stored locally first,
    then pushed to Central when network is available.
    """
    __tablename__ = "sync_queue"
    
    id = Column(Integer, primary_key=True, autoincrement=True)
    
    # What to sync
    entity_type = Column(String(50), nullable=False, index=True)  # 'result', 'audit_log'
    entity_id = Column(String(100), nullable=False)  # session_id, log_id, etc.
    
    # Payload to send
    payload = Column(JSON, nullable=False)
    
    # Sync status
    status = Column(Enum(SyncStatus), default=SyncStatus.PENDING, index=True)
    
    # Timing
    created_at = Column(DateTime, default=datetime.utcnow, index=True)
    last_attempt_at = Column(DateTime)
    completed_at = Column(DateTime)
    
    # Retry tracking
    retry_count = Column(Integer, default=0)
    max_retries = Column(Integer, default=10)
    last_error = Column(Text)
    
    # Deduplication
    __table_args__ = (
        # Prevent duplicate entries for same entity
        # (entity_type, entity_id) should be unique while pending
    )


class CentralConfig(Base):
    """
    Configuration for Central server connection.
    
    Stored locally to enable offline capability.
    """
    __tablename__ = "central_config"
    
    id = Column(Integer, primary_key=True, autoincrement=True)
    
    # Central server URL
    central_url = Column(String(500))  # e.g., https://central.company.com
    
    # Authentication
    api_key = Column(String(255))  # For tester-to-central auth
    
    # Sync settings
    sync_enabled = Column(Boolean, default=True)
    sync_interval_seconds = Column(Integer, default=30)
    
    # Last successful sync
    last_sync_at = Column(DateTime)
    last_sync_status = Column(String(50))
    
    # Central's public key for offline JWT validation
    central_public_key = Column(Text)
    
    updated_at = Column(DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)

