"""
SeqMaster Runtime - Barcode Scanner Driver
Layer: DRIVERS

Driver for USB barcode scanners (HID or Serial mode).
"""

import asyncio
from typing import Any, Dict, Optional

import structlog

from src.drivers.base import BaseDriver, DriverInfo, DriverResult, DriverStatus

logger = structlog.get_logger(__name__)


class BarcodeScannerDriver(BaseDriver):
    """
    Driver for USB barcode scanners.
    
    Supports scanners in:
    - HID keyboard mode (default)
    - Serial/COM mode
    """
    
    def __init__(self, device_id: str, 
                 port: Optional[str] = None,
                 mode: str = "serial",
                 config: Optional[Dict[str, Any]] = None):
        """
        Initialize barcode scanner driver.
        
        Args:
            device_id: Unique identifier
            port: Serial port for serial mode
            mode: "hid" or "serial"
            config: Additional configuration
        """
        super().__init__(device_id, config)
        self._port = port
        self._mode = mode
        self._serial = None
        self._last_barcode: Optional[str] = None
        self._read_buffer = ""
    
    @property
    def info(self) -> DriverInfo:
        return DriverInfo(
            name="Barcode Scanner Driver",
            version="1.0.0",
            driver_type="barcode_scanner",
            description="USB Barcode Scanner driver",
            capabilities=["scan", "read", "configure"]
        )
    
    async def connect(self) -> DriverResult:
        """Connect to barcode scanner."""
        try:
            if self._mode == "serial":
                import serial
                
                self._status = DriverStatus.CONNECTING
                
                self._serial = serial.Serial(
                    port=self._port,
                    baudrate=self._config.get("baudrate", 9600),
                    timeout=0.1
                )
                
                self._set_connected()
                
            elif self._mode == "hid":
                # HID mode - scanner acts as keyboard
                self._set_connected()
            
            logger.info("Barcode scanner connected", 
                       device_id=self._device_id,
                       mode=self._mode)
            
            return DriverResult(success=True)
            
        except Exception as e:
            self._set_error(str(e))
            logger.error("Barcode scanner connection failed", error=str(e))
            return DriverResult(success=False, error=str(e))
    
    async def disconnect(self) -> DriverResult:
        """Disconnect from barcode scanner."""
        try:
            if self._serial:
                self._serial.close()
                self._serial = None
            
            self._set_disconnected()
            return DriverResult(success=True)
            
        except Exception as e:
            return DriverResult(success=False, error=str(e))
    
    async def read(self, command: Optional[str] = None, 
                   timeout: float = 30.0, **kwargs) -> DriverResult:
        """
        Wait for and read barcode scan.
        
        Args:
            command: Not used
            timeout: Timeout in seconds (default 30s for manual scan)
        """
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        try:
            barcode = await self._wait_for_scan(timeout)
            
            if barcode:
                self._last_barcode = barcode
                return DriverResult(success=True, value=barcode)
            else:
                return DriverResult(success=False, error="Scan timeout")
                
        except Exception as e:
            return DriverResult(success=False, error=str(e))
    
    async def scan(self, timeout: float = 30.0) -> DriverResult:
        """
        Wait for barcode scan.
        
        Args:
            timeout: Timeout in seconds
            
        Returns:
            DriverResult with scanned barcode value
        """
        return await self.read(timeout=timeout)
    
    async def write(self, command: str, **kwargs) -> DriverResult:
        """Send command to scanner (for configuration)."""
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        if self._mode != "serial" or not self._serial:
            return DriverResult(success=False, error="Write only supported in serial mode")
        
        try:
            self._serial.write(command.encode())
            return DriverResult(success=True)
        except Exception as e:
            return DriverResult(success=False, error=str(e))
    
    async def reset(self) -> DriverResult:
        """Reset scanner."""
        self._last_barcode = None
        self._read_buffer = ""
        return DriverResult(success=True)
    
    async def selftest(self) -> DriverResult:
        """Scanner self-test."""
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        return DriverResult(success=True, value={"status": "ready"})
    
    async def _wait_for_scan(self, timeout: float) -> Optional[str]:
        """Wait for barcode scan."""
        if self._mode == "serial" and self._serial:
            start_time = asyncio.get_event_loop().time()
            
            while (asyncio.get_event_loop().time() - start_time) < timeout:
                if self._serial.in_waiting:
                    data = self._serial.read(self._serial.in_waiting)
                    self._read_buffer += data.decode('utf-8', errors='ignore')
                    
                    # Check for line ending (barcode complete)
                    if '\r' in self._read_buffer or '\n' in self._read_buffer:
                        barcode = self._read_buffer.strip()
                        self._read_buffer = ""
                        return barcode
                
                await asyncio.sleep(0.01)
            
            return None
        else:
            # HID mode - would need system-level keyboard capture
            await asyncio.sleep(timeout)
            return None
    
    @property
    def last_barcode(self) -> Optional[str]:
        """Get last scanned barcode."""
        return self._last_barcode
