"""
SeqMaster Runtime - Driver Base Interface
Layer: DRIVERS

Defines the abstract base interface for all hardware drivers.
All drivers must implement this interface.
"""

from abc import ABC, abstractmethod
from dataclasses import dataclass, field
from datetime import datetime
from enum import Enum
from typing import Any, Dict, List, Optional


class DriverStatus(str, Enum):
    """Driver connection status."""
    DISCONNECTED = "disconnected"
    CONNECTING = "connecting"
    CONNECTED = "connected"
    ERROR = "error"
    BUSY = "busy"


@dataclass
class DriverInfo:
    """Information about a driver."""
    name: str
    version: str
    driver_type: str
    description: str = ""
    author: str = "SeqMaster Team"
    capabilities: List[str] = field(default_factory=list)


@dataclass
class DriverResult:
    """Result from a driver operation."""
    success: bool
    value: Any = None
    error: Optional[str] = None
    timestamp: datetime = field(default_factory=datetime.now)
    raw_data: Optional[bytes] = None
    metadata: Dict[str, Any] = field(default_factory=dict)


class BaseDriver(ABC):
    """
    Abstract base class for all hardware drivers.
    
    All drivers must implement:
    - connect(): Establish connection to device
    - disconnect(): Close connection
    - read(): Read data from device
    - write(): Write data to device
    - reset(): Reset device to known state
    - selftest(): Run device self-test
    
    Optional overrides:
    - configure(): Configure device settings
    - get_status(): Get device status
    """
    
    def __init__(self, device_id: str, config: Optional[Dict[str, Any]] = None):
        """
        Initialize driver.
        
        Args:
            device_id: Unique identifier for the device
            config: Optional configuration dictionary
        """
        self._device_id = device_id
        self._config = config or {}
        self._status = DriverStatus.DISCONNECTED
        self._last_error: Optional[str] = None
        self._connected_at: Optional[datetime] = None
    
    @property
    @abstractmethod
    def info(self) -> DriverInfo:
        """Get driver information."""
        pass
    
    @property
    def device_id(self) -> str:
        """Get device ID."""
        return self._device_id
    
    @property
    def status(self) -> DriverStatus:
        """Get current status."""
        return self._status
    
    @property
    def is_connected(self) -> bool:
        """Check if connected."""
        return self._status == DriverStatus.CONNECTED
    
    @property
    def last_error(self) -> Optional[str]:
        """Get last error message."""
        return self._last_error
    
    @abstractmethod
    async def connect(self) -> DriverResult:
        """
        Connect to the device.
        
        Returns:
            DriverResult indicating success or failure
        """
        pass
    
    @abstractmethod
    async def disconnect(self) -> DriverResult:
        """
        Disconnect from the device.
        
        Returns:
            DriverResult indicating success or failure
        """
        pass
    
    @abstractmethod
    async def read(self, command: Optional[str] = None, **kwargs) -> DriverResult:
        """
        Read data from the device.
        
        Args:
            command: Optional command to send before reading
            **kwargs: Additional parameters
            
        Returns:
            DriverResult with read data
        """
        pass
    
    @abstractmethod
    async def write(self, command: str, **kwargs) -> DriverResult:
        """
        Write data to the device.
        
        Args:
            command: Command or data to write
            **kwargs: Additional parameters
            
        Returns:
            DriverResult indicating success or failure
        """
        pass
    
    @abstractmethod
    async def reset(self) -> DriverResult:
        """
        Reset device to known state.
        
        Returns:
            DriverResult indicating success or failure
        """
        pass
    
    @abstractmethod
    async def selftest(self) -> DriverResult:
        """
        Run device self-test.
        
        Returns:
            DriverResult with test results
        """
        pass
    
    async def configure(self, settings: Dict[str, Any]) -> DriverResult:
        """
        Configure device settings.
        
        Args:
            settings: Configuration dictionary
            
        Returns:
            DriverResult indicating success or failure
        """
        self._config.update(settings)
        return DriverResult(success=True, value=self._config)
    
    async def get_status(self) -> Dict[str, Any]:
        """
        Get device status.
        
        Returns:
            Status dictionary
        """
        return {
            "device_id": self._device_id,
            "status": self._status.value,
            "connected": self.is_connected,
            "last_error": self._last_error,
            "connected_at": self._connected_at.isoformat() if self._connected_at else None
        }
    
    def _set_connected(self) -> None:
        """Mark as connected."""
        self._status = DriverStatus.CONNECTED
        self._connected_at = datetime.now()
        self._last_error = None
    
    def _set_disconnected(self) -> None:
        """Mark as disconnected."""
        self._status = DriverStatus.DISCONNECTED
        self._connected_at = None
    
    def _set_error(self, error: str) -> None:
        """Mark as error state."""
        self._status = DriverStatus.ERROR
        self._last_error = error
