"""
SeqMaster Runtime - CAN Bus Driver
Layer: DRIVERS

Driver for USB-CAN adapters using python-can library.
"""

import asyncio
from typing import Any, Dict, List, Optional

import structlog

from src.drivers.base import BaseDriver, DriverInfo, DriverResult, DriverStatus

logger = structlog.get_logger(__name__)


class CANBusDriver(BaseDriver):
    """
    Driver for CAN bus communication via USB-CAN adapters.
    
    Supports:
    - CANable
    - PEAK PCAN-USB
    - Vector CANcase
    - SocketCAN (Linux)
    """
    
    def __init__(self, device_id: str, interface: str = "socketcan",
                 channel: str = "can0", bitrate: int = 500000,
                 config: Optional[Dict[str, Any]] = None):
        """
        Initialize CAN driver.
        
        Args:
            device_id: Unique identifier
            interface: CAN interface type (socketcan, pcan, vector, slcan)
            channel: CAN channel (can0, PCAN_USBBUS1, etc.)
            bitrate: CAN bitrate (default 500kbps)
            config: Additional configuration
        """
        super().__init__(device_id, config)
        self._interface = interface
        self._channel = channel
        self._bitrate = bitrate
        self._bus = None
        self._notifier = None
        self._receive_queue: asyncio.Queue = asyncio.Queue()
    
    @property
    def info(self) -> DriverInfo:
        return DriverInfo(
            name="CAN Bus Driver",
            version="1.0.0",
            driver_type="can",
            description="Driver for CAN bus communication",
            capabilities=["send", "receive", "filter", "reset"]
        )
    
    async def connect(self) -> DriverResult:
        """Connect to CAN bus."""
        try:
            import can
            
            self._status = DriverStatus.CONNECTING
            
            self._bus = can.interface.Bus(
                interface=self._interface,
                channel=self._channel,
                bitrate=self._bitrate
            )
            
            # Start receiver task
            asyncio.create_task(self._receive_loop())
            
            self._set_connected()
            logger.info("CAN bus connected", 
                       device_id=self._device_id,
                       interface=self._interface,
                       channel=self._channel)
            
            return DriverResult(success=True)
            
        except Exception as e:
            self._set_error(str(e))
            logger.error("CAN connection failed", error=str(e))
            return DriverResult(success=False, error=str(e))
    
    async def disconnect(self) -> DriverResult:
        """Disconnect from CAN bus."""
        try:
            if self._bus:
                self._bus.shutdown()
                self._bus = None
            
            self._set_disconnected()
            return DriverResult(success=True)
            
        except Exception as e:
            self._set_error(str(e))
            return DriverResult(success=False, error=str(e))
    
    async def read(self, command: Optional[str] = None, 
                   timeout: float = 1.0, **kwargs) -> DriverResult:
        """
        Read CAN message from queue.
        
        Args:
            command: Not used for CAN
            timeout: Read timeout in seconds
        """
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        try:
            msg = await asyncio.wait_for(
                self._receive_queue.get(),
                timeout=timeout
            )
            
            return DriverResult(
                success=True,
                value={
                    "arbitration_id": msg.arbitration_id,
                    "data": list(msg.data),
                    "timestamp": msg.timestamp,
                    "is_extended_id": msg.is_extended_id
                },
                raw_data=bytes(msg.data)
            )
            
        except asyncio.TimeoutError:
            return DriverResult(success=False, error="Timeout waiting for CAN message")
        except Exception as e:
            return DriverResult(success=False, error=str(e))
    
    async def write(self, command: str, **kwargs) -> DriverResult:
        """
        Send CAN message.
        
        Args:
            command: Not directly used - use send() method
        """
        return DriverResult(success=False, error="Use send() method for CAN")
    
    async def send(self, arbitration_id: int, data: List[int], 
                   extended_id: bool = False) -> DriverResult:
        """
        Send CAN message.
        
        Args:
            arbitration_id: CAN ID
            data: Data bytes (max 8)
            extended_id: Use extended ID format
        """
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        try:
            import can
            
            msg = can.Message(
                arbitration_id=arbitration_id,
                data=bytes(data),
                is_extended_id=extended_id
            )
            
            self._bus.send(msg)
            
            return DriverResult(
                success=True,
                value={"arbitration_id": arbitration_id, "data": data}
            )
            
        except Exception as e:
            logger.error("CAN send failed", error=str(e))
            return DriverResult(success=False, error=str(e))
    
    async def send_and_wait(self, arbitration_id: int, data: List[int],
                            response_id: int, timeout: float = 1.0) -> DriverResult:
        """
        Send CAN message and wait for response.
        
        Args:
            arbitration_id: Request CAN ID
            data: Request data
            response_id: Expected response CAN ID
            timeout: Response timeout
        """
        # Clear queue
        while not self._receive_queue.empty():
            try:
                self._receive_queue.get_nowait()
            except asyncio.QueueEmpty:
                break
        
        # Send request
        send_result = await self.send(arbitration_id, data)
        if not send_result.success:
            return send_result
        
        # Wait for response
        start_time = asyncio.get_event_loop().time()
        while (asyncio.get_event_loop().time() - start_time) < timeout:
            try:
                msg = await asyncio.wait_for(
                    self._receive_queue.get(),
                    timeout=timeout - (asyncio.get_event_loop().time() - start_time)
                )
                
                if msg.arbitration_id == response_id:
                    return DriverResult(
                        success=True,
                        value={
                            "arbitration_id": msg.arbitration_id,
                            "data": list(msg.data)
                        }
                    )
            except asyncio.TimeoutError:
                break
        
        return DriverResult(success=False, error=f"No response for ID 0x{response_id:X}")
    
    async def set_filter(self, can_id: int, mask: int = 0x7FF) -> DriverResult:
        """
        Set CAN filter.
        
        Args:
            can_id: CAN ID to accept
            mask: Filter mask
        """
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        try:
            self._bus.set_filters([{
                "can_id": can_id,
                "can_mask": mask,
                "extended": False
            }])
            return DriverResult(success=True)
        except Exception as e:
            return DriverResult(success=False, error=str(e))
    
    async def reset(self) -> DriverResult:
        """Reset CAN bus."""
        await self.disconnect()
        await asyncio.sleep(0.5)
        return await self.connect()
    
    async def selftest(self) -> DriverResult:
        """Self-test CAN bus."""
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        return DriverResult(success=True, value={"status": "ok"})
    
    async def _receive_loop(self) -> None:
        """Background receive loop."""
        while self.is_connected and self._bus:
            try:
                msg = self._bus.recv(timeout=0.1)
                if msg:
                    await self._receive_queue.put(msg)
            except Exception:
                break
