"""
SeqMaster Runtime - Driver Manager
Layer: DRIVERS

Manages driver lifecycle and provides access to hardware drivers.
"""

from typing import Any, Dict, List, Optional, Type

import structlog

from src.drivers.base import BaseDriver, DriverInfo
from src.drivers.scpi_driver import SCPIDriver, SCPIPSUDriver, SCPIOscilloscopeDriver
from src.drivers.can_driver import CANBusDriver
from src.drivers.relay_driver import RelayDriver
from src.drivers.barcode_driver import BarcodeScannerDriver
from src.drivers.mock_drivers import (
    MockSCPIDriver, MockCANDriver, MockRelayDriver, MockBarcodeDriver
)

logger = structlog.get_logger(__name__)


class DriverManager:
    """
    Central manager for hardware drivers.
    
    Provides:
    - Driver registration and lifecycle management
    - Driver lookup by ID or type
    - Mock driver injection for testing
    """
    
    def __init__(self, use_mocks: bool = False):
        """
        Initialize driver manager.
        
        Args:
            use_mocks: If True, use mock drivers instead of real hardware
        """
        self._drivers: Dict[str, BaseDriver] = {}
        self._use_mocks = use_mocks
        self._driver_registry: Dict[str, Type[BaseDriver]] = {
            "scpi": SCPIDriver,
            "scpi_psu": SCPIPSUDriver,
            "scpi_oscilloscope": SCPIOscilloscopeDriver,
            "can": CANBusDriver,
            "relay": RelayDriver,
            "barcode": BarcodeScannerDriver,
            # Mock drivers
            "mock_scpi": MockSCPIDriver,
            "mock_can": MockCANDriver,
            "mock_relay": MockRelayDriver,
            "mock_barcode": MockBarcodeDriver,
        }
    
    def register_driver(self, driver: BaseDriver) -> None:
        """
        Register a driver instance.
        
        Args:
            driver: Driver instance to register
        """
        self._drivers[driver.device_id] = driver
        logger.info("Driver registered", 
                   device_id=driver.device_id,
                   driver_type=driver.info.driver_type)
    
    def get_driver(self, device_id: str) -> Optional[BaseDriver]:
        """
        Get driver by device ID.
        
        Args:
            device_id: Device identifier
            
        Returns:
            Driver instance or None if not found
        """
        return self._drivers.get(device_id)
    
    def get_drivers_by_type(self, driver_type: str) -> List[BaseDriver]:
        """
        Get all drivers of a specific type.
        
        Args:
            driver_type: Type of driver (e.g., "scpi_psu", "can")
            
        Returns:
            List of matching drivers
        """
        return [
            d for d in self._drivers.values()
            if d.info.driver_type == driver_type
        ]
    
    def get_all_drivers(self) -> List[BaseDriver]:
        """Get all registered drivers."""
        return list(self._drivers.values())
    
    def create_driver(self, driver_type: str, device_id: str,
                      **kwargs) -> BaseDriver:
        """
        Create a new driver instance.
        
        Args:
            driver_type: Type of driver to create
            device_id: Unique device identifier
            **kwargs: Driver-specific configuration
            
        Returns:
            New driver instance
        """
        # Use mock if configured
        if self._use_mocks and not driver_type.startswith("mock_"):
            driver_type = f"mock_{driver_type}"
        
        driver_class = self._driver_registry.get(driver_type)
        if not driver_class:
            raise ValueError(f"Unknown driver type: {driver_type}")
        
        driver = driver_class(device_id=device_id, **kwargs)
        self.register_driver(driver)
        return driver
    
    async def connect_all(self) -> Dict[str, bool]:
        """
        Connect all registered drivers.
        
        Returns:
            Dictionary of device_id -> success status
        """
        results = {}
        for device_id, driver in self._drivers.items():
            try:
                result = await driver.connect()
                results[device_id] = result.success
            except Exception as e:
                logger.error("Driver connection failed", 
                           device_id=device_id, 
                           error=str(e))
                results[device_id] = False
        return results
    
    async def disconnect_all(self) -> None:
        """Disconnect all drivers."""
        for device_id, driver in self._drivers.items():
            try:
                await driver.disconnect()
            except Exception as e:
                logger.error("Driver disconnect failed",
                           device_id=device_id,
                           error=str(e))
    
    async def reset_all(self) -> None:
        """Reset all drivers to default state."""
        for device_id, driver in self._drivers.items():
            try:
                if driver.is_connected:
                    await driver.reset()
            except Exception as e:
                logger.error("Driver reset failed",
                           device_id=device_id,
                           error=str(e))
    
    def get_status(self) -> Dict[str, Dict[str, Any]]:
        """Get status of all drivers."""
        return {
            device_id: {
                "type": driver.info.driver_type,
                "name": driver.info.name,
                "status": driver.status.value,
                "connected": driver.is_connected,
                "last_error": driver.last_error
            }
            for device_id, driver in self._drivers.items()
        }
    
    def unregister_driver(self, device_id: str) -> bool:
        """
        Unregister a driver.
        
        Args:
            device_id: Device identifier
            
        Returns:
            True if driver was found and removed
        """
        if device_id in self._drivers:
            del self._drivers[device_id]
            return True
        return False


# Global driver manager instance
_driver_manager: Optional[DriverManager] = None
_driver_manager_lock = __import__('threading').Lock()


def get_driver_manager() -> DriverManager:
    """Get the global driver manager instance (thread-safe)."""
    global _driver_manager
    if _driver_manager is None:
        with _driver_manager_lock:
            if _driver_manager is None:  # Double-check locking
                _driver_manager = DriverManager()
    return _driver_manager


def init_driver_manager(use_mocks: bool = False) -> DriverManager:
    """Initialize the global driver manager."""
    global _driver_manager
    with _driver_manager_lock:
        if _driver_manager is not None:
            import structlog
            structlog.get_logger().warning("Reinitializing driver manager")
        _driver_manager = DriverManager(use_mocks=use_mocks)
    return _driver_manager
