"""
SeqMaster Runtime - Mock Drivers
Layer: DRIVERS

Mock implementations of all drivers for development and simulation.
"""

import asyncio
import random
from datetime import datetime
from typing import Any, Dict, List, Optional

import structlog

from src.drivers.base import BaseDriver, DriverInfo, DriverResult, DriverStatus

logger = structlog.get_logger(__name__)


class MockSCPIDriver(BaseDriver):
    """
    Mock SCPI driver for development and testing.
    Simulates a power supply with realistic behavior.
    """
    
    def __init__(self, device_id: str = "MOCK-PSU-001",
                 config: Optional[Dict[str, Any]] = None):
        super().__init__(device_id, config)
        self._voltage_setpoint = 0.0
        self._current_limit = 1.0
        self._output_enabled = False
        self._actual_voltage = 0.0
        self._actual_current = 0.0
    
    @property
    def info(self) -> DriverInfo:
        return DriverInfo(
            name="Mock SCPI Driver",
            version="1.0.0",
            driver_type="mock_scpi",
            description="Simulated SCPI power supply for testing",
            capabilities=["set_voltage", "set_current", "measure", "output_on", "output_off"]
        )
    
    async def connect(self) -> DriverResult:
        await asyncio.sleep(0.1)  # Simulate connection delay
        self._set_connected()
        logger.info("Mock SCPI device connected", device_id=self._device_id)
        return DriverResult(
            success=True,
            value={"idn": "MOCK,PSU-1000,SN001,1.0.0"}
        )
    
    async def disconnect(self) -> DriverResult:
        self._output_enabled = False
        self._set_disconnected()
        return DriverResult(success=True)
    
    async def read(self, command: Optional[str] = None, **kwargs) -> DriverResult:
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        await asyncio.sleep(0.02)  # Simulate query delay
        
        if command:
            cmd = command.upper()
            if "VOLT?" in cmd:
                return DriverResult(success=True, value=self._actual_voltage)
            elif "CURR?" in cmd:
                return DriverResult(success=True, value=self._actual_current)
            elif "IDN?" in cmd:
                return DriverResult(success=True, value="MOCK,PSU-1000,SN001,1.0.0")
        
        return DriverResult(success=True, value=None)
    
    async def write(self, command: str, **kwargs) -> DriverResult:
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        await asyncio.sleep(0.01)
        
        cmd = command.upper()
        
        if "VOLT" in cmd and "?" not in cmd:
            try:
                value = float(cmd.split()[-1].split(",")[0])
                self._voltage_setpoint = value
                if self._output_enabled:
                    self._actual_voltage = value + random.uniform(-0.01, 0.01)
            except:
                pass
        elif "CURR" in cmd and "?" not in cmd:
            try:
                value = float(cmd.split()[-1].split(",")[0])
                self._current_limit = value
            except:
                pass
        elif "OUTP ON" in cmd:
            self._output_enabled = True
            self._actual_voltage = self._voltage_setpoint + random.uniform(-0.01, 0.01)
            self._actual_current = random.uniform(0.01, 0.1)
        elif "OUTP OFF" in cmd:
            self._output_enabled = False
            self._actual_voltage = 0.0
            self._actual_current = 0.0
        
        return DriverResult(success=True)
    
    async def reset(self) -> DriverResult:
        self._voltage_setpoint = 0.0
        self._current_limit = 1.0
        self._output_enabled = False
        self._actual_voltage = 0.0
        self._actual_current = 0.0
        return DriverResult(success=True)
    
    async def selftest(self) -> DriverResult:
        return DriverResult(success=True, value={"passed": True, "result": "0"})
    
    # PSU convenience methods
    async def set_voltage(self, voltage: float, channel: int = 1) -> DriverResult:
        return await self.write(f"VOLT {voltage}")
    
    async def set_current(self, current: float, channel: int = 1) -> DriverResult:
        return await self.write(f"CURR {current}")
    
    async def output_on(self, channel: int = 1) -> DriverResult:
        return await self.write("OUTP ON")
    
    async def output_off(self, channel: int = 1) -> DriverResult:
        return await self.write("OUTP OFF")
    
    async def measure_voltage(self, channel: int = 1) -> DriverResult:
        return await self.read("MEAS:VOLT?")
    
    async def measure_current(self, channel: int = 1) -> DriverResult:
        return await self.read("MEAS:CURR?")


class MockCANDriver(BaseDriver):
    """Mock CAN bus driver for testing."""
    
    def __init__(self, device_id: str = "MOCK-CAN-001",
                 config: Optional[Dict[str, Any]] = None):
        super().__init__(device_id, config)
        self._message_queue: asyncio.Queue = asyncio.Queue()
        self._response_map: Dict[int, List[int]] = {}
    
    @property
    def info(self) -> DriverInfo:
        return DriverInfo(
            name="Mock CAN Driver",
            version="1.0.0",
            driver_type="mock_can",
            description="Simulated CAN bus for testing",
            capabilities=["send", "receive"]
        )
    
    async def connect(self) -> DriverResult:
        await asyncio.sleep(0.05)
        self._set_connected()
        return DriverResult(success=True)
    
    async def disconnect(self) -> DriverResult:
        self._set_disconnected()
        return DriverResult(success=True)
    
    async def read(self, command: Optional[str] = None, 
                   timeout: float = 1.0, **kwargs) -> DriverResult:
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        try:
            msg = await asyncio.wait_for(
                self._message_queue.get(),
                timeout=timeout
            )
            return DriverResult(success=True, value=msg)
        except asyncio.TimeoutError:
            return DriverResult(success=False, error="Timeout")
    
    async def write(self, command: str, **kwargs) -> DriverResult:
        return DriverResult(success=False, error="Use send() method")
    
    async def send(self, arbitration_id: int, data: List[int], 
                   extended_id: bool = False) -> DriverResult:
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        await asyncio.sleep(0.001)
        
        # Check for configured response
        if arbitration_id in self._response_map:
            response = {
                "arbitration_id": arbitration_id + 0x08,
                "data": self._response_map[arbitration_id],
                "timestamp": datetime.now().timestamp()
            }
            await self._message_queue.put(response)
        
        return DriverResult(success=True, value={"arbitration_id": arbitration_id, "data": data})
    
    def configure_response(self, request_id: int, response_data: List[int]) -> None:
        """Configure automatic response for testing."""
        self._response_map[request_id] = response_data
    
    async def reset(self) -> DriverResult:
        self._response_map.clear()
        while not self._message_queue.empty():
            try:
                self._message_queue.get_nowait()
            except asyncio.QueueEmpty:
                break
        return DriverResult(success=True)
    
    async def selftest(self) -> DriverResult:
        return DriverResult(success=True, value={"status": "ok"})


class MockRelayDriver(BaseDriver):
    """Mock relay driver for testing."""
    
    def __init__(self, device_id: str = "MOCK-RELAY-001",
                 num_channels: int = 8,
                 config: Optional[Dict[str, Any]] = None):
        super().__init__(device_id, config)
        self._num_channels = num_channels
        self._relay_states: List[bool] = [False] * num_channels
    
    @property
    def info(self) -> DriverInfo:
        return DriverInfo(
            name="Mock Relay Driver",
            version="1.0.0",
            driver_type="mock_relay",
            description=f"Simulated {self._num_channels}-channel relay module",
            capabilities=["set_relay", "get_relay", "set_all"]
        )
    
    @property
    def num_channels(self) -> int:
        return self._num_channels
    
    @property
    def relay_states(self) -> List[bool]:
        return self._relay_states.copy()
    
    async def connect(self) -> DriverResult:
        self._set_connected()
        return DriverResult(success=True)
    
    async def disconnect(self) -> DriverResult:
        self._relay_states = [False] * self._num_channels
        self._set_disconnected()
        return DriverResult(success=True)
    
    async def read(self, command: Optional[str] = None, **kwargs) -> DriverResult:
        return DriverResult(success=True, value=self._relay_states)
    
    async def write(self, command: str, **kwargs) -> DriverResult:
        return DriverResult(success=True)
    
    async def set_relay(self, channel: int, state: bool) -> DriverResult:
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        if channel < 1 or channel > self._num_channels:
            return DriverResult(success=False, error=f"Invalid channel {channel}")
        
        await asyncio.sleep(0.01)
        self._relay_states[channel - 1] = state
        return DriverResult(success=True, value={"channel": channel, "state": state})
    
    async def get_relay(self, channel: int) -> DriverResult:
        if channel < 1 or channel > self._num_channels:
            return DriverResult(success=False, error=f"Invalid channel {channel}")
        return DriverResult(success=True, value=self._relay_states[channel - 1])
    
    async def set_all(self, states: List[bool]) -> DriverResult:
        if len(states) != self._num_channels:
            return DriverResult(success=False, error="Invalid state count")
        self._relay_states = states.copy()
        return DriverResult(success=True, value=states)
    
    async def reset(self) -> DriverResult:
        self._relay_states = [False] * self._num_channels
        return DriverResult(success=True)
    
    async def selftest(self) -> DriverResult:
        return DriverResult(success=True, value={"tested_channels": self._num_channels})


class MockBarcodeDriver(BaseDriver):
    """Mock barcode scanner for testing."""
    
    def __init__(self, device_id: str = "MOCK-BARCODE-001",
                 config: Optional[Dict[str, Any]] = None):
        super().__init__(device_id, config)
        self._simulated_barcodes: List[str] = []
        self._last_barcode: Optional[str] = None
    
    @property
    def info(self) -> DriverInfo:
        return DriverInfo(
            name="Mock Barcode Scanner",
            version="1.0.0",
            driver_type="mock_barcode",
            description="Simulated barcode scanner for testing",
            capabilities=["scan"]
        )
    
    async def connect(self) -> DriverResult:
        self._set_connected()
        return DriverResult(success=True)
    
    async def disconnect(self) -> DriverResult:
        self._set_disconnected()
        return DriverResult(success=True)
    
    async def read(self, command: Optional[str] = None, 
                   timeout: float = 5.0, **kwargs) -> DriverResult:
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        if self._simulated_barcodes:
            await asyncio.sleep(0.5)  # Simulate scan delay
            barcode = self._simulated_barcodes.pop(0)
            self._last_barcode = barcode
            return DriverResult(success=True, value=barcode)
        else:
            await asyncio.sleep(timeout)
            return DriverResult(success=False, error="Timeout - no barcode scanned")
    
    async def scan(self, timeout: float = 5.0) -> DriverResult:
        return await self.read(timeout=timeout)
    
    async def write(self, command: str, **kwargs) -> DriverResult:
        return DriverResult(success=True)
    
    async def reset(self) -> DriverResult:
        self._simulated_barcodes.clear()
        self._last_barcode = None
        return DriverResult(success=True)
    
    async def selftest(self) -> DriverResult:
        return DriverResult(success=True, value={"status": "ready"})
    
    def simulate_scan(self, barcode: str) -> None:
        """Queue a barcode for simulated scanning."""
        self._simulated_barcodes.append(barcode)
    
    @property
    def last_barcode(self) -> Optional[str]:
        return self._last_barcode


# Driver factory for creating mock drivers
class MockDriverFactory:
    """Factory for creating mock drivers."""
    
    @staticmethod
    def create_psu(device_id: str = "MOCK-PSU-001") -> MockSCPIDriver:
        return MockSCPIDriver(device_id)
    
    @staticmethod
    def create_can(device_id: str = "MOCK-CAN-001") -> MockCANDriver:
        return MockCANDriver(device_id)
    
    @staticmethod
    def create_relay(device_id: str = "MOCK-RELAY-001", 
                     channels: int = 8) -> MockRelayDriver:
        return MockRelayDriver(device_id, channels)
    
    @staticmethod
    def create_barcode(device_id: str = "MOCK-BARCODE-001") -> MockBarcodeDriver:
        return MockBarcodeDriver(device_id)
