"""
SeqMaster Runtime - SCPI Driver
Layer: DRIVERS

Driver for SCPI-compatible instruments (PSU, Oscilloscope, DMM).
Uses PyVISA for communication.
"""

import asyncio
from typing import Any, Dict, Optional

import structlog

from src.drivers.base import BaseDriver, DriverInfo, DriverResult, DriverStatus

logger = structlog.get_logger(__name__)


class SCPIDriver(BaseDriver):
    """
    Driver for SCPI-compatible instruments.
    
    Supports:
    - Power supplies (PSU)
    - Oscilloscopes
    - Digital multimeters (DMM)
    - Any SCPI-compliant instrument
    """
    
    def __init__(self, device_id: str, resource_name: str, 
                 config: Optional[Dict[str, Any]] = None):
        """
        Initialize SCPI driver.
        
        Args:
            device_id: Unique identifier for this device
            resource_name: VISA resource name (e.g., "USB0::0x1AB1::0x04CE::DS1ZA...")
            config: Optional configuration
        """
        super().__init__(device_id, config)
        self._resource_name = resource_name
        self._instrument = None
        # Use centralized timeout constant
        from src.core.constants import SCPI_TIMEOUT_MS
        self._timeout = config.get("timeout", SCPI_TIMEOUT_MS) if config else SCPI_TIMEOUT_MS
        self._idn: Optional[str] = None
    
    @property
    def info(self) -> DriverInfo:
        return DriverInfo(
            name="SCPI Driver",
            version="1.0.0",
            driver_type="scpi",
            description="Driver for SCPI-compatible instruments",
            capabilities=["read", "write", "query", "reset", "selftest"]
        )
    
    async def connect(self) -> DriverResult:
        """Connect to the SCPI instrument."""
        try:
            import pyvisa
            
            self._status = DriverStatus.CONNECTING
            
            rm = pyvisa.ResourceManager('@py')
            self._instrument = rm.open_resource(
                self._resource_name,
                timeout=self._timeout
            )
            
            # Query identification
            self._idn = self._instrument.query("*IDN?").strip()
            
            self._set_connected()
            logger.info("SCPI device connected", 
                       device_id=self._device_id, 
                       idn=self._idn)
            
            return DriverResult(
                success=True,
                value={"idn": self._idn, "resource": self._resource_name}
            )
            
        except Exception as e:
            self._set_error(str(e))
            logger.error("SCPI connection failed", 
                        device_id=self._device_id, 
                        error=str(e))
            return DriverResult(success=False, error=str(e))
    
    async def disconnect(self) -> DriverResult:
        """Disconnect from the SCPI instrument."""
        try:
            if self._instrument:
                self._instrument.close()
                self._instrument = None
            
            self._set_disconnected()
            logger.info("SCPI device disconnected", device_id=self._device_id)
            return DriverResult(success=True)
            
        except Exception as e:
            self._set_error(str(e))
            return DriverResult(success=False, error=str(e))
    
    async def read(self, command: Optional[str] = None, **kwargs) -> DriverResult:
        """
        Read from the instrument (query).
        
        Args:
            command: SCPI query command (must end with ?)
            **kwargs: Additional parameters
        """
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        try:
            if command:
                response = self._instrument.query(command).strip()
            else:
                response = self._instrument.read().strip()
            
            # Try to parse numeric values
            try:
                value = float(response)
            except ValueError:
                value = response
            
            return DriverResult(success=True, value=value, raw_data=response.encode())
            
        except Exception as e:
            logger.error("SCPI read failed", 
                        device_id=self._device_id, 
                        command=command, 
                        error=str(e))
            return DriverResult(success=False, error=str(e))
    
    async def write(self, command: str, **kwargs) -> DriverResult:
        """
        Write to the instrument.
        
        Args:
            command: SCPI command to write
            **kwargs: Additional parameters
        """
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        try:
            self._instrument.write(command)
            return DriverResult(success=True, value=command)
            
        except Exception as e:
            logger.error("SCPI write failed", 
                        device_id=self._device_id, 
                        command=command, 
                        error=str(e))
            return DriverResult(success=False, error=str(e))
    
    async def query(self, command: str) -> DriverResult:
        """
        Query the instrument (write + read).
        
        Args:
            command: SCPI query command
        """
        return await self.read(command)
    
    async def reset(self) -> DriverResult:
        """Reset the instrument to default state."""
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        try:
            self._instrument.write("*RST")
            self._instrument.write("*CLS")
            await asyncio.sleep(0.5)
            return DriverResult(success=True)
            
        except Exception as e:
            self._set_error(str(e))
            return DriverResult(success=False, error=str(e))
    
    async def selftest(self) -> DriverResult:
        """Run instrument self-test."""
        if not self.is_connected:
            return DriverResult(success=False, error="Not connected")
        
        try:
            result = self._instrument.query("*TST?").strip()
            passed = result == "0"
            
            return DriverResult(
                success=True,
                value={"passed": passed, "result": result}
            )
            
        except Exception as e:
            return DriverResult(success=False, error=str(e))
    
    # PSU-specific methods
    async def set_voltage(self, voltage: float, channel: int = 1) -> DriverResult:
        """Set output voltage (PSU)."""
        return await self.write(f"VOLT {voltage},(@{channel})")
    
    async def set_current(self, current: float, channel: int = 1) -> DriverResult:
        """Set current limit (PSU)."""
        return await self.write(f"CURR {current},(@{channel})")
    
    async def output_on(self, channel: int = 1) -> DriverResult:
        """Enable output (PSU)."""
        return await self.write(f"OUTP ON,(@{channel})")
    
    async def output_off(self, channel: int = 1) -> DriverResult:
        """Disable output (PSU)."""
        return await self.write(f"OUTP OFF,(@{channel})")
    
    async def measure_voltage(self, channel: int = 1) -> DriverResult:
        """Measure actual voltage (PSU/DMM)."""
        return await self.query(f"MEAS:VOLT? (@{channel})")
    
    async def measure_current(self, channel: int = 1) -> DriverResult:
        """Measure actual current (PSU/DMM)."""
        return await self.query(f"MEAS:CURR? (@{channel})")


class SCPIPSUDriver(SCPIDriver):
    """Specialized driver for SCPI Power Supplies."""
    
    @property
    def info(self) -> DriverInfo:
        return DriverInfo(
            name="SCPI PSU Driver",
            version="1.0.0",
            driver_type="scpi_psu",
            description="Driver for SCPI-compatible power supplies",
            capabilities=[
                "set_voltage", "set_current", 
                "measure_voltage", "measure_current",
                "output_on", "output_off", "reset", "selftest"
            ]
        )


class SCPIOscilloscopeDriver(SCPIDriver):
    """Specialized driver for SCPI Oscilloscopes."""
    
    @property
    def info(self) -> DriverInfo:
        return DriverInfo(
            name="SCPI Oscilloscope Driver",
            version="1.0.0",
            driver_type="scpi_oscilloscope",
            description="Driver for SCPI-compatible oscilloscopes",
            capabilities=[
                "acquire", "measure", "set_timebase",
                "set_trigger", "reset", "selftest"
            ]
        )
    
    async def single_acquisition(self) -> DriverResult:
        """Perform single acquisition."""
        return await self.write(":SING")
    
    async def set_timebase(self, scale: float) -> DriverResult:
        """Set timebase scale (s/div)."""
        return await self.write(f":TIM:SCAL {scale}")
    
    async def set_channel_scale(self, channel: int, scale: float) -> DriverResult:
        """Set channel vertical scale (V/div)."""
        return await self.write(f":CHAN{channel}:SCAL {scale}")
    
    async def measure_frequency(self, channel: int = 1) -> DriverResult:
        """Measure frequency on channel."""
        return await self.query(f":MEAS:FREQ? CHAN{channel}")
    
    async def measure_vpp(self, channel: int = 1) -> DriverResult:
        """Measure peak-to-peak voltage."""
        return await self.query(f":MEAS:VPP? CHAN{channel}")
