"""
SeqMaster Runtime - Step Adapter Base
Layer: EXECUTOR

Abstract base class for step adapters.

Adapters handle execution of steps using different runtimes:
- PythonAdapter: Local Python plugins
- ExecAdapter: External executables with JSON I/O

Inspired by TestStand's module adapters (LabVIEW, .NET, C/C++, etc.)
"""

from abc import ABC, abstractmethod
from dataclasses import dataclass, field
from enum import Enum
from typing import Any, Callable, Dict, List, Optional, Type

import structlog

logger = structlog.get_logger(__name__)


class PortType(str, Enum):
    """Port data types."""
    NUMBER = "number"
    STRING = "string"
    BOOLEAN = "boolean"
    ARRAY = "array"
    OBJECT = "object"
    ANY = "any"


class PortDirection(str, Enum):
    """Port direction."""
    INPUT = "input"
    OUTPUT = "output"


@dataclass
class PortDef:
    """
    Definition of a step input/output port.
    
    Like TestStand's step parameters - defines the interface
    for passing data into and out of steps.
    """
    name: str
    type: PortType = PortType.ANY
    direction: PortDirection = PortDirection.INPUT
    required: bool = True
    default: Any = None
    description: str = ""
    
    def validate(self, value: Any) -> bool:
        """Check if value matches port type."""
        if value is None:
            return not self.required
        
        if self.type == PortType.ANY:
            return True
        elif self.type == PortType.NUMBER:
            return isinstance(value, (int, float))
        elif self.type == PortType.STRING:
            return isinstance(value, str)
        elif self.type == PortType.BOOLEAN:
            return isinstance(value, bool)
        elif self.type == PortType.ARRAY:
            return isinstance(value, list)
        elif self.type == PortType.OBJECT:
            return isinstance(value, dict)
        
        return True


@dataclass
class StepSchema:
    """
    Schema for a step type.
    
    Defines the inputs and outputs of a step, like a function signature.
    Used for validation, documentation, and UI generation.
    """
    name: str
    description: str = ""
    inputs: List[PortDef] = field(default_factory=list)
    outputs: List[PortDef] = field(default_factory=list)
    version: str = "1.0"
    author: str = ""
    tags: List[str] = field(default_factory=list)
    
    def get_input(self, name: str) -> Optional[PortDef]:
        """Get input port by name."""
        for port in self.inputs:
            if port.name == name:
                return port
        return None
    
    def get_output(self, name: str) -> Optional[PortDef]:
        """Get output port by name."""
        for port in self.outputs:
            if port.name == name:
                return port
        return None
    
    def validate_inputs(self, inputs: Dict[str, Any]) -> List[str]:
        """
        Validate inputs against schema.
        
        Returns list of validation errors (empty if valid).
        """
        errors = []
        
        for port in self.inputs:
            if port.name not in inputs:
                if port.required and port.default is None:
                    errors.append(f"Missing required input: {port.name}")
            else:
                if not port.validate(inputs[port.name]):
                    errors.append(
                        f"Invalid type for {port.name}: expected {port.type.value}"
                    )
        
        return errors
    
    def apply_defaults(self, inputs: Dict[str, Any]) -> Dict[str, Any]:
        """Apply default values to missing inputs."""
        result = inputs.copy()
        
        for port in self.inputs:
            if port.name not in result and port.default is not None:
                result[port.name] = port.default
        
        return result


@dataclass
class StepResult:
    """
    Result from executing a step.
    
    Contains output values, status, and any errors.
    """
    success: bool
    outputs: Dict[str, Any] = field(default_factory=dict)
    error: Optional[str] = None
    error_code: Optional[str] = None
    duration_ms: float = 0.0
    metadata: Dict[str, Any] = field(default_factory=dict)


class StepAdapter(ABC):
    """
    Abstract base class for step adapters.
    
    Adapters are responsible for:
    1. Loading step modules (plugins, executables, etc.)
    2. Getting step schemas (input/output definitions)
    3. Executing steps with given inputs
    4. Returning outputs
    
    Usage:
        adapter = PythonAdapter()
        
        # Get schema
        schema = await adapter.get_schema("math", "add")
        
        # Execute
        result = await adapter.execute("math", "add", {"a": 1, "b": 2})
        # result.outputs = {"return": 3}
    """
    
    @property
    @abstractmethod
    def name(self) -> str:
        """Adapter name (e.g., 'python', 'exec')."""
        pass
    
    @abstractmethod
    async def get_schema(self, module: str, method: str) -> Optional[StepSchema]:
        """
        Get schema for a step.
        
        Args:
            module: Module/plugin name or path
            method: Method/function name
            
        Returns:
            StepSchema or None if not found
        """
        pass
    
    @abstractmethod
    async def execute(
        self, 
        module: str, 
        method: str, 
        inputs: Dict[str, Any]
    ) -> StepResult:
        """
        Execute a step.
        
        Args:
            module: Module/plugin name or path
            method: Method/function name
            inputs: Input values (already resolved from expressions)
            
        Returns:
            StepResult with outputs
        """
        pass
    
    async def validate_inputs(
        self, 
        module: str, 
        method: str, 
        inputs: Dict[str, Any]
    ) -> List[str]:
        """
        Validate inputs before execution.
        
        Returns list of validation errors.
        """
        schema = await self.get_schema(module, method)
        if not schema:
            return [f"Schema not found for {module}.{method}"]
        return schema.validate_inputs(inputs)
    
    def is_available(self) -> bool:
        """Check if adapter is available and properly configured."""
        return True


class AdapterManager:
    """
    Manager for step adapters.
    
    Provides a registry of adapters and lookup by name.
    
    Usage:
        manager = AdapterManager()
        manager.register(PythonAdapter())
        manager.register(ExecAdapter())
        
        # Get adapter by name
        adapter = manager.get("python")
        
        # Execute step
        result = await adapter.execute("math", "add", {"a": 1, "b": 2})
    """
    
    def __init__(self):
        """Initialize adapter manager."""
        self._adapters: Dict[str, StepAdapter] = {}
    
    def register(self, adapter: StepAdapter) -> None:
        """Register an adapter."""
        self._adapters[adapter.name] = adapter
        logger.info("Registered adapter", name=adapter.name)
    
    def get(self, name: str) -> Optional[StepAdapter]:
        """Get adapter by name."""
        return self._adapters.get(name)
    
    def get_all(self) -> List[StepAdapter]:
        """Get all registered adapters."""
        return list(self._adapters.values())
    
    def get_available(self) -> List[StepAdapter]:
        """Get all available adapters."""
        return [a for a in self._adapters.values() if a.is_available()]
    
    async def execute(
        self, 
        adapter_name: str, 
        module: str, 
        method: str, 
        inputs: Dict[str, Any]
    ) -> StepResult:
        """
        Execute a step using specified adapter.
        
        Convenience method that looks up adapter and executes.
        """
        adapter = self.get(adapter_name)
        if not adapter:
            return StepResult(
                success=False,
                error=f"Adapter not found: {adapter_name}",
                error_code="ADAPTER_NOT_FOUND"
            )
        
        return await adapter.execute(module, method, inputs)


# Global adapter manager instance
_adapter_manager: Optional[AdapterManager] = None


def get_adapter_manager() -> AdapterManager:
    """Get global adapter manager instance."""
    global _adapter_manager
    if _adapter_manager is None:
        _adapter_manager = AdapterManager()
    return _adapter_manager


def register_adapter(adapter: StepAdapter) -> None:
    """Register adapter with global manager."""
    get_adapter_manager().register(adapter)
