"""
SeqMaster Runtime - Compare Adapter
Layer: EXECUTOR

Simple adapter for comparing a variable value against limits.
Used for steps that only need to validate a value without calling external code.

Usage in sequence:
    {
        "adapter": "compare",
        "inputs": {
            "value": "${Locals.voltage}",
            "expected": 5.0    # Optional for exact match
        },
        "limits": {
            "lower": 4.5,
            "upper": 5.5
        }
    }
"""

from typing import Any, Dict, Optional

import structlog

from src.executor.adapters.base import (
    StepAdapter, StepSchema, StepResult, PortDef, PortType, PortDirection
)

logger = structlog.get_logger(__name__)


class CompareAdapter(StepAdapter):
    """
    Adapter for comparing values against limits or expected values.
    
    This adapter doesn't execute external code - it just validates
    a value that was already computed or read from a variable.
    
    Inputs:
        - value: The value to validate (required)
        - expected: Expected value for exact match (optional)
        
    The step's limits are applied by the executor, not this adapter.
    This adapter simply returns the value for limit checking.
    """
    
    @property
    def name(self) -> str:
        return "compare"
    
    async def get_schema(self, module: str, method: str) -> Optional[StepSchema]:
        """
        Get schema for compare operations.
        
        For compare adapter, module and method are ignored - 
        there's only one operation: compare the input value.
        """
        return StepSchema(
            name="compare",
            description="Compare a value against limits or expected value",
            inputs=[
                PortDef(
                    name="value",
                    type=PortType.ANY,
                    direction=PortDirection.INPUT,
                    required=True,
                    description="The value to validate"
                ),
                PortDef(
                    name="expected",
                    type=PortType.ANY,
                    direction=PortDirection.INPUT,
                    required=False,
                    description="Expected value for exact match comparison"
                ),
                PortDef(
                    name="tolerance",
                    type=PortType.NUMBER,
                    direction=PortDirection.INPUT,
                    required=False,
                    default=0,
                    description="Tolerance for numeric comparison"
                ),
            ],
            outputs=[
                PortDef(
                    name="return",
                    type=PortType.ANY,
                    direction=PortDirection.OUTPUT,
                    description="The input value (for limit checking by executor)"
                ),
                PortDef(
                    name="passed",
                    type=PortType.BOOLEAN,
                    direction=PortDirection.OUTPUT,
                    description="Whether the comparison passed"
                ),
            ],
            version="1.0",
            author="SeqMaster",
            tags=["compare", "validate", "limits"]
        )
    
    async def execute(
        self, 
        module: str, 
        method: str, 
        inputs: Dict[str, Any]
    ) -> StepResult:
        """
        Execute comparison.
        
        If 'expected' is provided, does exact match (with optional tolerance).
        Otherwise, just returns the value for limit checking by executor.
        """
        value = inputs.get("value")
        expected = inputs.get("expected")
        tolerance = inputs.get("tolerance", 0)
        case_sensitive = inputs.get("case_sensitive", True)
        
        logger.debug("Compare adapter executing", 
                    value=value, 
                    expected=expected, 
                    tolerance=tolerance,
                    case_sensitive=case_sensitive)
        
        passed = True
        error = None
        
        # If expected value is provided, do exact match comparison
        if expected is not None:
            if isinstance(value, (int, float)) and isinstance(expected, (int, float)):
                # Numeric comparison with tolerance
                diff = abs(float(value) - float(expected))
                passed = diff <= tolerance
                if not passed:
                    error = f"Value {value} differs from expected {expected} by {diff} (tolerance: {tolerance})"
            elif isinstance(value, str) and isinstance(expected, str):
                # String comparison with case sensitivity option
                if case_sensitive:
                    passed = value == expected
                else:
                    passed = value.lower() == expected.lower()
                if not passed:
                    error = f"Value '{value}' does not match expected '{expected}'"
            else:
                # Exact match for other types (convert to string for comparison)
                passed = str(value) == str(expected)
                if not passed:
                    error = f"Value '{value}' does not match expected '{expected}'"
        
        # Note: Limit checking (lower/upper) is done by the executor, not this adapter
        # This adapter just returns the value for the executor to validate
        
        return StepResult(
            success=True,  # Adapter succeeded, even if comparison failed
            outputs={
                "return": value,
                "passed": passed,
                "value": value,
                "expected": expected,
            },
            error=error if not passed else None,
            metadata={
                "comparison_passed": passed,
                "had_expected": expected is not None,
            }
        )
    
    async def validate_inputs(self, inputs: Dict[str, Any]) -> bool:
        """Validate that required inputs are present."""
        return "value" in inputs
    
    async def cleanup(self) -> None:
        """No cleanup needed for compare adapter."""
        pass
