"""
SeqMaster Runtime - Exec Adapter
Layer: EXECUTOR

Adapter for executing external executables as steps.

Executables communicate via JSON on stdin/stdout:
- Input: JSON object written to stdin
- Output: JSON object read from stdout
- Errors: Non-zero exit code or JSON with "error" field

This allows any language (C#, Go, Rust, Python scripts, etc.) to be used
for step implementation.

Example C# executable:
    // Program.cs
    var input = JsonSerializer.Deserialize<Input>(Console.In.ReadToEnd());
    var result = Process(input);
    Console.WriteLine(JsonSerializer.Serialize(result));

Example usage in sequence:
    steps:
      - id: M001
        adapter: exec
        executable: "C:/Tools/db_lookup.exe"
        inputs:
          serial_number: "Locals.serial"
        outputs:
          cal_offset: "Locals.calibration.offset"
          cal_gain: "Locals.calibration.gain"
"""

import asyncio
import json
import os
import time
from pathlib import Path
from typing import Any, Dict, List, Optional

import structlog

from src.executor.adapters.base import (
    StepAdapter,
    StepSchema,
    StepResult,
    PortDef,
    PortType,
    PortDirection,
)

logger = structlog.get_logger(__name__)


class ExecAdapter(StepAdapter):
    """
    Adapter for external executables.
    
    Features:
    - JSON-based input/output communication
    - Timeout handling
    - Working directory support
    - Environment variable injection
    - Schema files (optional .schema.json alongside executable)
    """
    
    def __init__(
        self, 
        default_timeout_ms: int = 30000,
        tools_dir: Optional[Path] = None
    ):
        """
        Initialize Exec adapter.
        
        Args:
            default_timeout_ms: Default timeout for executions
            tools_dir: Optional directory for relative executable paths
        """
        self._default_timeout_ms = default_timeout_ms
        self._tools_dir = tools_dir
        self._schema_cache: Dict[str, StepSchema] = {}
        
        logger.info("Exec adapter initialized",
                   default_timeout_ms=default_timeout_ms,
                   tools_dir=str(tools_dir) if tools_dir else None)
    
    @property
    def name(self) -> str:
        return "exec"
    
    async def get_schema(self, module: str, method: str) -> Optional[StepSchema]:
        """
        Get schema for an executable.
        
        Looks for a .schema.json file alongside the executable.
        If not found, returns a generic schema.
        
        Args:
            module: Executable path
            method: Ignored for exec adapter (executable is the module)
        """
        cache_key = module
        if cache_key in self._schema_cache:
            return self._schema_cache[cache_key]
        
        # Try to find schema file
        exe_path = self._resolve_path(module)
        if exe_path:
            schema_path = exe_path.with_suffix(".schema.json")
            if schema_path.exists():
                try:
                    schema = self._load_schema_file(schema_path)
                    self._schema_cache[cache_key] = schema
                    return schema
                except Exception as e:
                    logger.warning("Failed to load schema file",
                                 path=str(schema_path),
                                 error=str(e))
        
        # Return generic schema
        schema = StepSchema(
            name=Path(module).stem if module else "unknown",
            description=f"External executable: {module}",
            inputs=[],  # Unknown inputs
            outputs=[
                PortDef(
                    name="return",
                    type=PortType.OBJECT,
                    direction=PortDirection.OUTPUT,
                    description="All outputs from executable"
                )
            ]
        )
        self._schema_cache[cache_key] = schema
        return schema
    
    async def execute(
        self, 
        module: str, 
        method: str, 
        inputs: Dict[str, Any],
        timeout_ms: Optional[int] = None,
        working_dir: Optional[str] = None,
        env: Optional[Dict[str, str]] = None
    ) -> StepResult:
        """
        Execute an external executable.
        
        Args:
            module: Executable path (absolute or relative to tools_dir)
            method: Ignored for exec adapter
            inputs: Input data to send as JSON to stdin
            timeout_ms: Optional timeout override
            working_dir: Optional working directory
            env: Optional environment variables to add
        """
        start_time = time.time()
        timeout = (timeout_ms or self._default_timeout_ms) / 1000.0
        
        # Resolve executable path
        exe_path = self._resolve_path(module)
        if not exe_path or not exe_path.exists():
            return StepResult(
                success=False,
                error=f"Executable not found: {module}",
                error_code="EXECUTABLE_NOT_FOUND"
            )
        
        # Prepare environment
        process_env = os.environ.copy()
        if env:
            process_env.update(env)
        
        # Prepare working directory
        cwd = working_dir or (str(exe_path.parent) if exe_path else None)
        
        try:
            # Serialize inputs to JSON
            input_json = json.dumps(inputs, default=str)
            
            logger.debug("Executing external process",
                        executable=str(exe_path),
                        inputs=inputs)
            
            # Create process
            process = await asyncio.create_subprocess_exec(
                str(exe_path),
                stdin=asyncio.subprocess.PIPE,
                stdout=asyncio.subprocess.PIPE,
                stderr=asyncio.subprocess.PIPE,
                cwd=cwd,
                env=process_env
            )
            
            # Communicate with timeout
            try:
                stdout, stderr = await asyncio.wait_for(
                    process.communicate(input_json.encode("utf-8")),
                    timeout=timeout
                )
            except asyncio.TimeoutError:
                process.kill()
                await process.wait()
                return StepResult(
                    success=False,
                    error=f"Execution timed out after {timeout}s",
                    error_code="TIMEOUT",
                    duration_ms=(time.time() - start_time) * 1000
                )
            
            duration_ms = (time.time() - start_time) * 1000
            
            # Check exit code
            if process.returncode != 0:
                error_msg = stderr.decode("utf-8").strip() if stderr else f"Exit code: {process.returncode}"
                return StepResult(
                    success=False,
                    error=error_msg,
                    error_code=f"EXIT_{process.returncode}",
                    duration_ms=duration_ms
                )
            
            # Parse output JSON
            stdout_str = stdout.decode("utf-8").strip()
            if not stdout_str:
                # Empty output is valid
                return StepResult(
                    success=True,
                    outputs={"return": None},
                    duration_ms=duration_ms
                )
            
            try:
                output_data = json.loads(stdout_str)
            except json.JSONDecodeError as e:
                return StepResult(
                    success=False,
                    error=f"Invalid JSON output: {e}",
                    error_code="INVALID_OUTPUT",
                    duration_ms=duration_ms,
                    metadata={"stdout": stdout_str[:500]}
                )
            
            # Check for error in output
            if isinstance(output_data, dict) and "error" in output_data:
                return StepResult(
                    success=False,
                    error=output_data["error"],
                    error_code=output_data.get("error_code", "EXEC_ERROR"),
                    outputs=output_data,
                    duration_ms=duration_ms
                )
            
            # Build outputs
            if isinstance(output_data, dict):
                outputs = output_data
            else:
                outputs = {"return": output_data}
            
            return StepResult(
                success=True,
                outputs=outputs,
                duration_ms=duration_ms
            )
            
        except Exception as e:
            duration_ms = (time.time() - start_time) * 1000
            logger.error("Exec execution failed",
                        executable=str(exe_path),
                        error=str(e))
            
            return StepResult(
                success=False,
                error=str(e),
                error_code="EXECUTION_ERROR",
                duration_ms=duration_ms
            )
    
    def _resolve_path(self, path_str: str) -> Optional[Path]:
        """Resolve executable path."""
        if not path_str:
            return None
        
        path = Path(path_str)
        
        # Absolute path
        if path.is_absolute():
            return path
        
        # Relative to tools_dir
        if self._tools_dir:
            full_path = self._tools_dir / path
            if full_path.exists():
                return full_path
        
        # Relative to current directory
        if path.exists():
            return path.resolve()
        
        # Try to find in PATH
        import shutil
        which_result = shutil.which(path_str)
        if which_result:
            return Path(which_result)
        
        return path  # Return as-is, will fail in execute
    
    def _load_schema_file(self, schema_path: Path) -> StepSchema:
        """Load schema from JSON file."""
        with open(schema_path, "r") as f:
            data = json.load(f)
        
        inputs = []
        for inp in data.get("inputs", []):
            inputs.append(PortDef(
                name=inp["name"],
                type=PortType(inp.get("type", "any")),
                direction=PortDirection.INPUT,
                required=inp.get("required", True),
                default=inp.get("default"),
                description=inp.get("description", "")
            ))
        
        outputs = []
        for out in data.get("outputs", []):
            outputs.append(PortDef(
                name=out["name"],
                type=PortType(out.get("type", "any")),
                direction=PortDirection.OUTPUT,
                description=out.get("description", "")
            ))
        
        return StepSchema(
            name=data.get("name", schema_path.stem),
            description=data.get("description", ""),
            inputs=inputs,
            outputs=outputs,
            version=data.get("version", "1.0"),
            author=data.get("author", "")
        )


# Example schema file format:
# db_lookup.schema.json
# {
#     "name": "db_lookup",
#     "description": "Look up calibration data from database",
#     "version": "1.0",
#     "inputs": [
#         {
#             "name": "serial_number",
#             "type": "string",
#             "required": true,
#             "description": "DUT serial number"
#         }
#     ],
#     "outputs": [
#         {
#             "name": "cal_offset",
#             "type": "number",
#             "description": "Calibration offset"
#         },
#         {
#             "name": "cal_gain",
#             "type": "number",
#             "description": "Calibration gain"
#         },
#         {
#             "name": "cal_date",
#             "type": "string",
#             "description": "Calibration date"
#         }
#     ]
# }
