"""
SeqMaster Runtime - Python Adapter
Layer: EXECUTOR

Adapter for executing Python plugins as steps.

Plugins are Python modules with functions that can be called as steps.
The adapter automatically introspects functions to generate schemas
from type hints and docstrings.

Plugin location: runtime/src/executor/plugins/

Example plugin:
    # plugins/math_ops.py
    def add(a: float, b: float) -> float:
        '''Add two numbers.'''
        return a + b
    
    def average(values: list) -> float:
        '''Calculate average of values.'''
        return sum(values) / len(values) if values else 0.0

Example usage in sequence:
    steps:
      - id: M001
        adapter: python
        plugin: math_ops
        method: add
        inputs:
          a: 10.5
          b: "Locals.offset"
        outputs:
          return: "Locals.result"
"""

import asyncio
import importlib
import importlib.util
import inspect
import sys
import time
from pathlib import Path
from typing import Any, Callable, Dict, List, Optional, get_type_hints

import structlog

from src.executor.adapters.base import (
    StepAdapter,
    StepSchema,
    StepResult,
    PortDef,
    PortType,
    PortDirection,
)

logger = structlog.get_logger(__name__)


def _python_type_to_port_type(python_type: type) -> PortType:
    """Convert Python type hint to PortType."""
    if python_type in (int, float):
        return PortType.NUMBER
    elif python_type == str:
        return PortType.STRING
    elif python_type == bool:
        return PortType.BOOLEAN
    elif python_type == list:
        return PortType.ARRAY
    elif python_type == dict:
        return PortType.OBJECT
    else:
        return PortType.ANY


class PythonAdapter(StepAdapter):
    """
    Adapter for Python plugins.
    
    Features:
    - Auto-discovery of plugins from plugins directory
    - Schema generation from type hints and docstrings
    - Sync and async function support
    - Driver wrapper integration
    """
    
    def __init__(self, plugins_dir: Optional[Path] = None):
        """
        Initialize Python adapter.
        
        Args:
            plugins_dir: Directory containing plugin modules
        """
        self._plugins_dir = plugins_dir or Path(__file__).parent.parent / "plugins"
        self._loaded_modules: Dict[str, Any] = {}
        self._schema_cache: Dict[str, StepSchema] = {}
        
        # Ensure plugins dir exists
        self._plugins_dir.mkdir(parents=True, exist_ok=True)
        
        # Add plugins dir to path if not already
        plugins_str = str(self._plugins_dir)
        if plugins_str not in sys.path:
            sys.path.insert(0, plugins_str)
        
        logger.info("Python adapter initialized", plugins_dir=str(self._plugins_dir))
    
    @property
    def name(self) -> str:
        return "python"
    
    async def get_schema(self, module: str, method: str) -> Optional[StepSchema]:
        """
        Get schema for a Python function.
        
        Introspects the function to extract:
        - Parameters as inputs (from signature and type hints)
        - Return type as output
        - Description from docstring
        """
        cache_key = f"{module}.{method}"
        if cache_key in self._schema_cache:
            return self._schema_cache[cache_key]
        
        # Load module
        mod = self._load_module(module)
        if mod is None:
            return None
        
        # Get function
        func = getattr(mod, method, None)
        if func is None or not callable(func):
            logger.warning("Method not found", module=module, method=method)
            return None
        
        # Build schema from function signature
        schema = self._build_schema_from_function(func, method)
        self._schema_cache[cache_key] = schema
        
        return schema
    
    async def execute(
        self, 
        module: str, 
        method: str, 
        inputs: Dict[str, Any]
    ) -> StepResult:
        """
        Execute a Python function.
        
        Supports both sync and async functions.
        """
        start_time = time.time()
        
        try:
            # Load module
            mod = self._load_module(module)
            if mod is None:
                return StepResult(
                    success=False,
                    error=f"Module not found: {module}",
                    error_code="MODULE_NOT_FOUND"
                )
            
            # Get function
            func = getattr(mod, method, None)
            if func is None or not callable(func):
                return StepResult(
                    success=False,
                    error=f"Method not found: {method}",
                    error_code="METHOD_NOT_FOUND"
                )
            
            # Get schema and apply defaults
            schema = await self.get_schema(module, method)
            if schema:
                inputs = schema.apply_defaults(inputs)
            
            # Call function
            if asyncio.iscoroutinefunction(func):
                result = await func(**inputs)
            else:
                # Run sync function in executor to not block
                loop = asyncio.get_event_loop()
                result = await loop.run_in_executor(None, lambda: func(**inputs))
            
            # Build outputs
            outputs = self._build_outputs(func, result)
            
            duration_ms = (time.time() - start_time) * 1000
            
            return StepResult(
                success=True,
                outputs=outputs,
                duration_ms=duration_ms
            )
            
        except Exception as e:
            duration_ms = (time.time() - start_time) * 1000
            logger.error("Python execution failed",
                        module=module,
                        method=method,
                        error=str(e))
            
            return StepResult(
                success=False,
                error=str(e),
                error_code="EXECUTION_ERROR",
                duration_ms=duration_ms
            )
    
    def _load_module(self, module_name: str) -> Optional[Any]:
        """Load a plugin module by name."""
        if module_name in self._loaded_modules:
            return self._loaded_modules[module_name]
        
        try:
            # Try direct import first (for installed packages)
            mod = importlib.import_module(module_name)
            self._loaded_modules[module_name] = mod
            logger.debug("Loaded module", module=module_name)
            return mod
        except ImportError:
            pass
        
        # Try loading from plugins directory
        module_path = self._plugins_dir / f"{module_name}.py"
        if module_path.exists():
            try:
                spec = importlib.util.spec_from_file_location(module_name, module_path)
                if spec and spec.loader:
                    mod = importlib.util.module_from_spec(spec)
                    sys.modules[module_name] = mod
                    spec.loader.exec_module(mod)
                    self._loaded_modules[module_name] = mod
                    logger.debug("Loaded module from file", 
                               module=module_name, 
                               path=str(module_path))
                    return mod
            except Exception as e:
                logger.error("Failed to load module", 
                           module=module_name, 
                           error=str(e))
        
        logger.warning("Module not found", module=module_name)
        return None
    
    def _build_schema_from_function(self, func: Callable, method_name: str) -> StepSchema:
        """Build StepSchema from function signature."""
        inputs = []
        outputs = []
        
        # Get signature
        sig = inspect.signature(func)
        
        # Get type hints
        try:
            hints = get_type_hints(func)
        except Exception:
            hints = {}
        
        # Build input ports from parameters
        for param_name, param in sig.parameters.items():
            if param_name in ("self", "cls"):
                continue
            
            # Determine type
            if param_name in hints:
                port_type = _python_type_to_port_type(hints[param_name])
            else:
                port_type = PortType.ANY
            
            # Determine if required and default
            required = param.default == inspect.Parameter.empty
            default = None if required else param.default
            
            inputs.append(PortDef(
                name=param_name,
                type=port_type,
                direction=PortDirection.INPUT,
                required=required,
                default=default
            ))
        
        # Build output port from return type
        return_hint = hints.get("return")
        if return_hint and return_hint != type(None):
            port_type = _python_type_to_port_type(return_hint)
            outputs.append(PortDef(
                name="return",
                type=port_type,
                direction=PortDirection.OUTPUT
            ))
        else:
            # Always have a return output
            outputs.append(PortDef(
                name="return",
                type=PortType.ANY,
                direction=PortDirection.OUTPUT
            ))
        
        # Get description from docstring
        description = inspect.getdoc(func) or ""
        
        return StepSchema(
            name=method_name,
            description=description,
            inputs=inputs,
            outputs=outputs
        )
    
    def _build_outputs(self, func: Callable, result: Any) -> Dict[str, Any]:
        """Build outputs dict from function result."""
        # Always include result under "return" key for standard binding
        # If result is a dict, also include its keys as individual outputs
        outputs = {"return": result}
        
        # If result is a dict, add each key as a separate output
        # This allows binding individual properties OR the whole dict
        if isinstance(result, dict):
            for key, value in result.items():
                outputs[key] = value
        
        return outputs
    
    def get_available_plugins(self) -> List[str]:
        """Get list of available plugin modules."""
        plugins = []
        
        if self._plugins_dir.exists():
            for path in self._plugins_dir.glob("*.py"):
                if not path.name.startswith("_"):
                    plugins.append(path.stem)
        
        return plugins
    
    def reload_module(self, module_name: str) -> bool:
        """Reload a plugin module."""
        if module_name in self._loaded_modules:
            try:
                mod = self._loaded_modules[module_name]
                importlib.reload(mod)
                logger.info("Reloaded module", module=module_name)
                
                # Clear schema cache for this module
                keys_to_remove = [k for k in self._schema_cache if k.startswith(f"{module_name}.")]
                for key in keys_to_remove:
                    del self._schema_cache[key]
                
                return True
            except Exception as e:
                logger.error("Failed to reload module", module=module_name, error=str(e))
                return False
        return False
