"""
SeqMaster Runtime - Sequence Models
Layer: EXECUTOR

Pydantic models for test sequences and property sets.
"""

from datetime import datetime
from enum import Enum
from typing import Any, Dict, List, Optional, Union

from pydantic import BaseModel, Field


class StepType(str, Enum):
    """Type of step in a sequence."""
    TEST = "test"           # Measurement/validation step
    ACTION = "action"       # Action without validation
    FLOW = "flow"          # Flow control (loop, conditional)
    GROUP = "group"        # Test group container
    
    # TestStand-style flow control step types
    LABEL = "label"         # Named target for GOTO
    GOTO = "goto"           # Jump to a label
    STATEMENT = "statement" # Execute expression and assign to variable
    IF = "if"               # Conditional branch start
    ELSE = "else"           # Else branch
    ELSE_IF = "else_if"     # Else-if branch
    END_IF = "end_if"       # End conditional block
    FOR_EACH = "for_each"   # For-each loop
    WHILE = "while"         # While loop
    END_LOOP = "end_loop"   # End loop block
    BREAK = "break"         # Exit current loop
    CONTINUE = "continue"   # Skip to next loop iteration
    MESSAGE = "message"     # Display message to operator (like Message Popup)
    DELAY = "delay"         # Pause execution for specified time


class ComparisonOperator(str, Enum):
    """Comparison operators for validation."""
    EQ = "eq"               # Equal
    NE = "ne"               # Not equal
    GT = "gt"               # Greater than
    GE = "ge"               # Greater than or equal
    LT = "lt"               # Less than
    LE = "le"               # Less than or equal
    BETWEEN = "between"     # Between min and max
    CONTAINS = "contains"   # String contains
    MATCHES = "matches"     # Regex match


class ConditionOperator(str, Enum):
    """Logical operators for conditions."""
    AND = "and"
    OR = "or"
    NOT = "not"


# ============================================
# STEP DEFINITIONS
# ============================================

class StepParameter(BaseModel):
    """Parameter for a step."""
    name: str
    value: Any
    description: Optional[str] = None


class StepLimit(BaseModel):
    """Limits for validation."""
    # Numeric limits
    lower: Optional[float] = None
    upper: Optional[float] = None
    expected: Optional[Any] = None
    tolerance: Optional[float] = None
    comparison: ComparisonOperator = ComparisonOperator.BETWEEN
    unit: Optional[str] = None
    
    # String limits
    expected_string: Optional[str] = None
    case_sensitive: bool = True
    
    # Limit type
    type: Optional[str] = None  # 'numeric', 'string', 'boolean', etc.


class StepCondition(BaseModel):
    """Condition for conditional execution."""
    variable: str
    operator: ComparisonOperator
    value: Any
    logic: Optional[ConditionOperator] = None
    next_condition: Optional["StepCondition"] = None


class RetryConfig(BaseModel):
    """Retry configuration."""
    max_retries: int = 3
    delay_ms: int = 1000
    backoff_multiplier: float = 1.0


class LoopConfig(BaseModel):
    """Loop configuration."""
    count: Optional[int] = None
    variable: Optional[str] = None
    start: Optional[int] = 0
    end: Optional[int] = None
    step: Optional[int] = 1
    until_condition: Optional[StepCondition] = None


class ErrorCode(BaseModel):
    """Error code definition for step failures."""
    id: str = Field(..., description="Unique error code identifier")
    title: str = Field(..., description="Short error title")
    description: str = Field(default="", description="Detailed error description")
    image: Optional[str] = Field(None, description="Path or URL to error image")
    range_min: Optional[float] = Field(None, description="Min value for auto-selection")
    range_max: Optional[float] = Field(None, description="Max value for auto-selection")


class TestStep(BaseModel):
    """Individual test step definition."""
    id: str = Field(..., description="Step ID (M### for measurement, TS### for group)")
    name: str
    type: StepType = StepType.TEST
    description: Optional[str] = None
    
    # Adapter-based execution (new system)
    adapter: Optional[str] = Field(
        None, 
        description="Adapter type: 'python' or 'exec'"
    )
    plugin: Optional[str] = Field(
        None, 
        description="Plugin module name (for python adapter)"
    )
    executable: Optional[str] = Field(
        None, 
        description="Executable path (for exec adapter)"
    )
    method: Optional[str] = Field(
        None, 
        description="Method/function name to call"
    )
    inputs: Dict[str, Any] = Field(
        default_factory=dict,
        description="Input bindings: {port_name: value_or_expression}"
    )
    outputs: Dict[str, str] = Field(
        default_factory=dict,
        description="Output bindings: {port_name: variable_reference}"
    )
    
    # Compare adapter - variable to compare against limits
    compare_variable: Optional[str] = Field(
        None,
        description="Variable expression to compare (for compare adapter)"
    )
    
    # Hardware/Driver (legacy system, still supported)
    driver: Optional[str] = None
    driver_method: Optional[str] = None
    
    # Parameters (legacy, use inputs instead)
    parameters: List[StepParameter] = Field(default_factory=list)
    
    # Limits and validation
    limits: Optional[StepLimit] = None
    property_ref: Optional[str] = None  # Reference to property set
    
    # Timing
    timeout_ms: int = 30000
    delay_before_ms: int = 0
    delay_after_ms: int = 0
    
    # Flow control - critical abort for safety-critical steps
    critical_abort: bool = Field(
        default=False, 
        description="If true, abort immediately on failure without user interaction and run cleanup"
    )
    retry: Optional[RetryConfig] = None
    
    # Conditional execution
    condition: Optional[StepCondition] = None
    skip_if: Optional[str] = None  # Expression to evaluate
    
    # Loop (for flow type)
    loop: Optional[LoopConfig] = None
    
    # Nested steps (for groups)
    steps: Optional[List["TestStep"]] = None
    
    # Flow control (TestStand-style)
    label: Optional[str] = Field(
        None, 
        description="Label name for LABEL step or target for GOTO step"
    )
    expression: Optional[str] = Field(
        None,
        description="Expression to evaluate for STATEMENT, IF, WHILE steps"
    )
    target_variable: Optional[str] = Field(
        None,
        description="Variable to assign result for STATEMENT step"
    )
    message_text: Optional[str] = Field(
        None,
        description="Message text for MESSAGE step (supports expressions)"
    )
    message_buttons: Optional[List[str]] = Field(
        None,
        description="Buttons for MESSAGE step (e.g., ['OK', 'Cancel'])"
    )
    for_each_variable: Optional[str] = Field(
        None,
        description="Variable name for FOR_EACH iterator"
    )
    for_each_collection: Optional[str] = Field(
        None,
        description="Collection expression for FOR_EACH loop"
    )
    
    # Metadata
    tags: List[str] = Field(default_factory=list)
    metadata: Dict[str, Any] = Field(default_factory=dict)
    
    # Operator interaction
    requires_operator_input: bool = False
    operator_prompt: Optional[str] = None
    
    # Error codes for failure diagnosis
    error_codes: List[ErrorCode] = Field(
        default_factory=list, 
        description="Possible error codes for this step, shown to operator on failure"
    )
    
    # Disabled/Skip
    disabled: bool = Field(default=False, description="If true, step is skipped during execution")


class TestGroup(BaseModel):
    """Test group (TS###) containing multiple steps."""
    id: str = Field(..., description="Group ID (TS###)")
    name: str
    description: Optional[str] = None
    steps: List[TestStep] = Field(default_factory=list)
    
    # Callback steps (TestStand-style)
    retry_setup_steps: List[TestStep] = Field(
        default_factory=list, 
        description="Steps run before retrying this group - restore tester state"
    )
    
    # Group-level settings
    condition: Optional[StepCondition] = None
    loop: Optional[LoopConfig] = None
    
    # Disabled/Skip
    disabled: bool = Field(default=False, description="If true, group is skipped during execution")
    
    # Operator interaction on failure
    allow_retry: bool = Field(
        default=False, 
        description="If true, operator can retry entire group on step failure"
    )
    allow_continue: bool = Field(
        default=False,
        description="If true, operator can continue to next step on failure"
    )
    
    # Metadata
    tags: List[str] = Field(default_factory=list)


# ============================================
# SEQUENCE DEFINITION
# ============================================

class SequenceMetadata(BaseModel):
    """Metadata for a sequence."""
    author: Optional[str] = None
    created_at: Optional[datetime] = None
    modified_at: Optional[datetime] = None
    approved_by: Optional[str] = None
    approval_date: Optional[datetime] = None
    change_notes: Optional[str] = None
    tags: List[str] = Field(default_factory=list)


class SequenceRequirements(BaseModel):
    """Hardware requirements for a sequence."""
    capabilities: Dict[str, Any] = Field(default_factory=dict)
    drivers: List[str] = Field(default_factory=list)
    min_runtime_version: Optional[str] = None


class TestSequence(BaseModel):
    """Complete test sequence definition."""
    id: str
    version: str
    name: str
    description: Optional[str] = None
    
    # DUT information
    dut_type: Optional[str] = None
    dut_revisions: List[str] = Field(default_factory=list)
    
    # Custom container type definitions
    types: Dict[str, Dict[str, Any]] = Field(
        default_factory=dict,
        description="Custom container type definitions"
    )
    
    # Variable definitions per scope
    variables: Dict[str, Dict[str, Any]] = Field(
        default_factory=dict,
        description="Variable definitions: {scope: {name: {type, default, ...}}}"
    )
    
    # Requirements
    requirements: Optional[SequenceRequirements] = None
    property_set_ref: Optional[str] = None
    
    # Setup and teardown
    setup_steps: List[TestStep] = Field(default_factory=list)
    teardown_steps: List[TestStep] = Field(default_factory=list)
    
    # Cleanup callback - runs before any retry or on abort (global)
    cleanup_steps: List[TestStep] = Field(
        default_factory=list,
        description="Global cleanup steps run before retry or on abort - clears signals/supplies"
    )
    
    # Result callbacks - run based on final sequence result
    passed_steps: List[TestStep] = Field(
        default_factory=list,
        description="Steps run when sequence passes (e.g., green LED, success sound, label print)"
    )
    failed_steps: List[TestStep] = Field(
        default_factory=list,
        description="Steps run when sequence fails (e.g., red LED, error sound, rework routing)"
    )
    
    # Main test sequence
    groups: List[TestGroup] = Field(default_factory=list)
    steps: List[TestStep] = Field(default_factory=list)  # Ungrouped steps
    
    # Flow control
    abort_on_first_failure: bool = True
    continue_on_group_fail: bool = False
    
    # Metadata
    metadata: SequenceMetadata = Field(default_factory=SequenceMetadata)
    
    def get_all_steps(self) -> List[TestStep]:
        """Get all steps in order (flattened)."""
        all_steps = []
        all_steps.extend(self.setup_steps)
        
        for group in self.groups:
            all_steps.extend(group.steps)
        
        all_steps.extend(self.steps)
        all_steps.extend(self.teardown_steps)
        
        return all_steps
    
    def get_total_executions(self) -> int:
        """
        Get total number of ACTION step executions including loops.
        
        Only counts TEST and ACTION step types - excludes flow control steps
        (if, while, delay, message, goto, etc.) for accurate progress display.
        
        Unlike get_all_steps() which returns unique step definitions,
        this accounts for loops and returns actual execution count.
        """
        # Flow control step types that should NOT be counted
        flow_control_types = {
            StepType.LABEL, StepType.GOTO, StepType.STATEMENT,
            StepType.IF, StepType.ELSE, StepType.ELSE_IF, StepType.END_IF,
            StepType.FOR_EACH, StepType.WHILE, StepType.END_LOOP,
            StepType.BREAK, StepType.CONTINUE, StepType.MESSAGE, StepType.DELAY
        }
        
        def count_action_steps(steps: list) -> int:
            return sum(1 for s in steps if s.type not in flow_control_types)
        
        total = 0
        total += count_action_steps(self.setup_steps)
        
        for group in self.groups:
            loop_count = 1
            if group.loop and group.loop.count:
                loop_count = group.loop.count
            total += count_action_steps(group.steps) * loop_count
        
        total += count_action_steps(self.steps)
        total += count_action_steps(self.teardown_steps)
        
        return total


# ============================================
# PROPERTY SET DEFINITION
# ============================================

class PropertyValue(BaseModel):
    """Single property value with limits."""
    name: str
    lower: Optional[float] = None
    upper: Optional[float] = None
    nominal: Optional[float] = None
    unit: Optional[str] = None
    description: Optional[str] = None


class PropertySet(BaseModel):
    """Property set (limits) for a DUT type/revision."""
    id: str
    version: str
    name: str
    
    dut_type: str
    dut_revision: Optional[str] = None
    
    properties: Dict[str, PropertyValue] = Field(default_factory=dict)
    
    metadata: Dict[str, Any] = Field(default_factory=dict)
    
    def get_limits(self, property_name: str) -> Optional[PropertyValue]:
        """Get limits for a property."""
        return self.properties.get(property_name)


# Enable forward references
TestStep.model_rebuild()
StepCondition.model_rebuild()
