"""
SeqMaster Runtime - Built-in Plugin
Layer: EXECUTOR/PLUGINS

Built-in utility functions for test sequences.

These functions provide common operations that don't require
external hardware or libraries.
"""

import time
from datetime import datetime
from typing import Any, Dict, List, Optional


# ============================================
# COMPARISON FUNCTIONS
# ============================================

def compare(value: float, min_val: float, max_val: float) -> bool:
    """
    Check if value is within range.
    
    Returns True if min_val <= value <= max_val.
    """
    return min_val <= value <= max_val


def equals(a: Any, b: Any) -> bool:
    """Check if two values are equal."""
    return a == b


def not_equals(a: Any, b: Any) -> bool:
    """Check if two values are not equal."""
    return a != b


def greater_than(a: float, b: float) -> bool:
    """Check if a > b."""
    return a > b


def less_than(a: float, b: float) -> bool:
    """Check if a < b."""
    return a < b


def is_null(value: Any) -> bool:
    """Check if value is None/null."""
    return value is None


def is_not_null(value: Any) -> bool:
    """Check if value is not None/null."""
    return value is not None


def pass_value(value: Any) -> Any:
    """
    Pass through a value unchanged.
    
    Useful for:
    - Assigning one variable to another
    - Appending to arrays: outputs.return -> "Locals.readings[]"
    - Extracting object properties
    """
    return value


# ============================================
# STRING FUNCTIONS
# ============================================

def string_contains(text: str, substring: str) -> bool:
    """Check if text contains substring."""
    return substring in text


def string_starts_with(text: str, prefix: str) -> bool:
    """Check if text starts with prefix."""
    return text.startswith(prefix)


def string_ends_with(text: str, suffix: str) -> bool:
    """Check if text ends with suffix."""
    return text.endswith(suffix)


def string_format(template: str, **kwargs) -> str:
    """
    Format a string template with values.
    
    Example: string_format("{serial}-{date}", serial="DUT001", date="2025-01-25")
    """
    return template.format(**kwargs)


def string_concat(*parts) -> str:
    """Concatenate multiple strings."""
    return "".join(str(p) for p in parts)


def string_length(text: str) -> int:
    """Get length of string."""
    return len(text)


# ============================================
# DELAY FUNCTIONS
# ============================================

def delay_ms(milliseconds: int) -> bool:
    """
    Pause execution for specified milliseconds.
    
    Returns True when complete.
    """
    # Convert to number if string (from sequence editor)
    if isinstance(milliseconds, str):
        milliseconds = float(milliseconds)
    time.sleep(milliseconds / 1000.0)
    return True


def delay_seconds(seconds: float) -> bool:
    """
    Pause execution for specified seconds.
    
    Returns True when complete.
    """
    # Convert to number if string (from sequence editor)
    if isinstance(seconds, str):
        seconds = float(seconds)
    time.sleep(seconds)
    return True


# ============================================
# DATE/TIME FUNCTIONS
# ============================================

def get_timestamp() -> str:
    """Get current timestamp in ISO format."""
    return datetime.now().isoformat()


def get_date() -> str:
    """Get current date (YYYY-MM-DD)."""
    return datetime.now().strftime("%Y-%m-%d")


def get_time() -> str:
    """Get current time (HH:MM:SS)."""
    return datetime.now().strftime("%H:%M:%S")


def get_unix_timestamp() -> float:
    """Get current Unix timestamp."""
    return time.time()


# ============================================
# ARRAY FUNCTIONS
# ============================================

def array_length(arr: list) -> int:
    """Get length of array."""
    return len(arr) if arr else 0


def array_get(arr: list, index: int, default: Any = None) -> Any:
    """
    Get element at index, with optional default.
    
    Returns default if index is out of bounds.
    """
    try:
        return arr[index]
    except (IndexError, TypeError):
        return default


def array_first(arr: list, default: Any = None) -> Any:
    """Get first element of array."""
    return arr[0] if arr else default


def array_last(arr: list, default: Any = None) -> Any:
    """Get last element of array."""
    return arr[-1] if arr else default


def array_slice(arr: list, start: int, end: int = None) -> list:
    """
    Get a slice of the array.
    
    Equivalent to arr[start:end].
    """
    return arr[start:end]


def array_reverse(arr: list) -> list:
    """Return reversed copy of array."""
    return list(reversed(arr))


def array_sort(arr: list, descending: bool = False) -> list:
    """Return sorted copy of array."""
    return sorted(arr, reverse=descending)


def array_unique(arr: list) -> list:
    """Return array with duplicates removed (preserves order)."""
    seen = set()
    result = []
    for item in arr:
        if item not in seen:
            seen.add(item)
            result.append(item)
    return result


# ============================================
# LOGIC FUNCTIONS
# ============================================

def all_true(*values) -> bool:
    """Check if all values are truthy."""
    return all(values)


def any_true(*values) -> bool:
    """Check if any value is truthy."""
    return any(values)


def if_then_else(condition: bool, if_true: Any, if_false: Any) -> Any:
    """
    Conditional expression.
    
    Returns if_true when condition is True, else if_false.
    """
    return if_true if condition else if_false


def coalesce(*values) -> Any:
    """
    Return first non-None value.
    
    Returns None if all values are None.
    """
    for v in values:
        if v is not None:
            return v
    return None


# ============================================
# PASS/FAIL HELPERS
# ============================================

def pass_result(message: str = "PASS") -> Dict[str, Any]:
    """Return a passing result."""
    return {
        "passed": True,
        "message": message
    }


def fail_result(message: str = "FAIL") -> Dict[str, Any]:
    """Return a failing result."""
    return {
        "passed": False,
        "message": message
    }


def evaluate_limit(
    value: float,
    lower: Optional[float] = None,
    upper: Optional[float] = None,
    nominal: Optional[float] = None,
    tolerance: Optional[float] = None
) -> Dict[str, Any]:
    """
    Evaluate a measured value against limits.
    
    Supports:
    - Range check: lower <= value <= upper
    - Tolerance check: |value - nominal| <= tolerance
    
    Returns dict with: passed, value, lower, upper, message
    """
    passed = True
    message = "PASS"
    
    # Calculate effective limits
    effective_lower = lower
    effective_upper = upper
    
    if nominal is not None and tolerance is not None:
        effective_lower = nominal - tolerance
        effective_upper = nominal + tolerance
    
    # Check limits
    if effective_lower is not None and value < effective_lower:
        passed = False
        message = f"Value {value} below lower limit {effective_lower}"
    elif effective_upper is not None and value > effective_upper:
        passed = False
        message = f"Value {value} above upper limit {effective_upper}"
    
    return {
        "passed": passed,
        "value": value,
        "lower": effective_lower,
        "upper": effective_upper,
        "message": message
    }
