"""
SeqMaster Demo Plugin
Layer: EXECUTOR/PLUGINS

Demonstrates how Python plugins work with the variable system.
Each function can:
- Accept inputs (resolved from variables or literals)
- Return single values or dicts (for multiple outputs)
"""

import random
import time
from datetime import datetime
from typing import Dict, List, Any


def scan_barcode() -> str:
    """
    Simulate barcode scanning.
    
    Returns:
        Scanned serial number
    """
    # Simulate scan delay
    time.sleep(0.1)
    
    # Generate fake serial
    timestamp = datetime.now().strftime("%y%m%d")
    random_part = random.randint(1000, 9999)
    return f"DUT-{timestamp}-{random_part}"


def log_message(message: str, tester: str = "", serial: str = "") -> bool:
    """
    Log a message (simulated).
    
    Args:
        message: Message to log
        tester: Tester ID
        serial: DUT serial number
    
    Returns:
        Success status
    """
    full_message = f"[{tester}] {message}"
    if serial:
        full_message += f" - Serial: {serial}"
    
    print(f"LOG: {full_message}")
    return True


def measure_voltage(channel: int = 1, noise: float = 0.1) -> float:
    """
    Simulate voltage measurement.
    
    Args:
        channel: ADC channel (1-4)
        noise: Random noise amplitude
    
    Returns:
        Measured voltage
    """
    # Simulate measurement time
    time.sleep(0.05)
    
    # Base voltage with some noise
    base_voltage = 12.0
    measured = base_voltage + random.uniform(-noise, noise)
    
    return round(measured, 3)


def measure_current(channel: int = 1) -> float:
    """
    Simulate current measurement.
    
    Returns:
        Measured current in Amps
    """
    time.sleep(0.05)
    return round(0.5 + random.uniform(-0.05, 0.05), 4)


def measure_all(channel: int = 1) -> Dict[str, Any]:
    """
    Measure multiple values at once.
    
    Returns:
        Dict with voltage, current, power, timestamp
    """
    voltage = measure_voltage(channel)
    current = measure_current(channel)
    power = voltage * current
    
    return {
        "voltage": voltage,
        "current": current,
        "power": round(power, 3),
        "timestamp": datetime.now().isoformat(),
        "channel": channel
    }


def check_tolerance(value: float, target: float, tolerance_percent: float) -> bool:
    """
    Check if value is within tolerance of target.
    
    Args:
        value: Measured value
        target: Target value
        tolerance_percent: Tolerance in percent (e.g., 5.0 for ±5%)
    
    Returns:
        True if within tolerance
    """
    tolerance = target * (tolerance_percent / 100.0)
    lower = target - tolerance
    upper = target + tolerance
    
    result = lower <= value <= upper
    print(f"Tolerance check: {value:.3f} in [{lower:.3f}, {upper:.3f}] = {result}")
    
    return result


def increment(value: float) -> float:
    """
    Increment a counter by 1.
    
    Args:
        value: Current counter value
    
    Returns:
        Incremented value
    """
    return value + 1


def generate_report(
    serial: str,
    voltage: float,
    current: float,
    passed: bool,
    readings: List[float]
) -> Dict[str, Any]:
    """
    Generate a test report.
    
    Args:
        serial: DUT serial number
        voltage: Final voltage reading
        current: Final current reading
        passed: Overall pass/fail
        readings: All voltage readings
    
    Returns:
        Report dict
    """
    avg_reading = sum(readings) / len(readings) if readings else 0.0
    
    report = {
        "serial": serial,
        "timestamp": datetime.now().isoformat(),
        "result": "PASS" if passed else "FAIL",
        "measurements": {
            "voltage": voltage,
            "current": current,
            "power": round(voltage * current, 3)
        },
        "statistics": {
            "readings_count": len(readings),
            "average_voltage": round(avg_reading, 3),
            "min_voltage": round(min(readings), 3) if readings else 0,
            "max_voltage": round(max(readings), 3) if readings else 0
        }
    }
    
    print(f"\n{'='*50}")
    print(f"TEST REPORT - {serial}")
    print(f"{'='*50}")
    print(f"Result: {report['result']}")
    print(f"Voltage: {voltage:.3f}V, Current: {current:.4f}A")
    print(f"Readings: {readings}")
    print(f"Average: {avg_reading:.3f}V")
    print(f"{'='*50}\n")
    
    return report


def pass_value(value: Any) -> Any:
    """
    Pass through a value (useful for variable assignment).
    
    Args:
        value: Any value to pass through
    
    Returns:
        The same value
    """
    return value


# =============================================================================
# Copyright © 2026 Haris Hujic. All rights reserved.
# =============================================================================
