"""
SeqMaster Runtime - Math Operations Plugin
Layer: EXECUTOR/PLUGINS

Mathematical operations for use in test sequences.

All functions are designed to be called as steps via the Python adapter.
Type hints are used to generate step schemas automatically.
"""

from typing import List, Optional, Union

Number = Union[int, float]


def add(a: float, b: float) -> float:
    """Add two numbers."""
    return a + b


def subtract(a: float, b: float) -> float:
    """Subtract b from a."""
    return a - b


def multiply(a: float, b: float) -> float:
    """Multiply two numbers."""
    return a * b


def divide(a: float, b: float) -> float:
    """
    Divide a by b.
    
    Raises ZeroDivisionError if b is zero.
    """
    if b == 0:
        raise ZeroDivisionError("Cannot divide by zero")
    return a / b


def average(values: list) -> float:
    """
    Calculate the average (mean) of a list of numbers.
    
    Returns 0.0 for empty list.
    """
    if not values:
        return 0.0
    return sum(values) / len(values)


def sum_values(values: list) -> float:
    """Calculate the sum of a list of numbers."""
    return sum(values)


def min_value(values: list) -> float:
    """Find the minimum value in a list."""
    if not values:
        return 0.0
    return min(values)


def max_value(values: list) -> float:
    """Find the maximum value in a list."""
    if not values:
        return 0.0
    return max(values)


def clamp(value: float, min_val: float, max_val: float) -> float:
    """Clamp a value between min and max."""
    return max(min_val, min(max_val, value))


def scale(value: float, gain: float, offset: float = 0.0) -> float:
    """
    Apply linear scaling: result = value * gain + offset.
    
    Useful for calibration adjustments.
    """
    return value * gain + offset


def percentage(value: float, total: float) -> float:
    """Calculate percentage: (value / total) * 100."""
    if total == 0:
        return 0.0
    return (value / total) * 100.0


def in_range(value: float, lower: float, upper: float) -> bool:
    """Check if value is within range [lower, upper]."""
    return lower <= value <= upper


def tolerance_check(
    value: float, 
    nominal: float, 
    tolerance: float, 
    is_percentage: bool = False
) -> bool:
    """
    Check if value is within tolerance of nominal.
    
    Args:
        value: Measured value
        nominal: Expected nominal value
        tolerance: Tolerance (absolute or percentage)
        is_percentage: If True, tolerance is a percentage
        
    Returns:
        True if within tolerance
    """
    if is_percentage:
        allowed = abs(nominal) * (tolerance / 100.0)
    else:
        allowed = tolerance
    
    return abs(value - nominal) <= allowed


def linear_interpolate(
    x: float,
    x1: float, y1: float,
    x2: float, y2: float
) -> float:
    """
    Linear interpolation between two points.
    
    Given points (x1, y1) and (x2, y2), find y for given x.
    """
    if x2 == x1:
        return y1
    return y1 + (y2 - y1) * (x - x1) / (x2 - x1)


def standard_deviation(values: list) -> float:
    """
    Calculate the sample standard deviation.
    
    Returns 0.0 for empty or single-element list.
    """
    n = len(values)
    if n < 2:
        return 0.0
    
    mean = sum(values) / n
    variance = sum((x - mean) ** 2 for x in values) / (n - 1)
    return variance ** 0.5
