"""
SeqMaster Runtime - Variable Store
Layer: EXECUTOR

Manages variable storage with scopes, inspired by NI TestStand.

Scopes (in order of lookup):
1. Step - Current step properties (read-only)
2. Locals - Sequence-local variables
3. Parameters - Sequence input/output parameters
4. Station - Persistent station-wide variables
5. RunState - Runtime state (read-only)

Features:
- Dot-notation access: store.get("Locals.calibration.offset")
- Array indexing: store.get("Locals.readings[0]")
- Array append: store.set("Locals.readings[]", 12.5)
- Type validation on set
- Station scope persistence to JSON
"""

import json
from pathlib import Path
from typing import Any, Dict, List, Optional, Set

import structlog

from src.executor.variables.types import (
    PropertyType,
    VariableDef,
    ContainerDef,
    VariableScope,
    VariableReference,
)


class DotDict:
    """
    Dict wrapper that allows attribute-style access for expression evaluation.
    
    Enables expressions like `Locals.voltage` instead of `Locals["voltage"]`.
    Nested dicts are also wrapped recursively.
    
    Uses __slots__ for memory efficiency.
    """
    __slots__ = ('_data', '_cache')
    
    def __init__(self, data: Dict[str, Any]):
        object.__setattr__(self, '_data', data)
        object.__setattr__(self, '_cache', {})  # Cache for nested DotDict wrappers
    
    def __getattr__(self, name: str) -> Any:
        if name.startswith('_'):
            return object.__getattribute__(self, name)
        
        data = object.__getattribute__(self, '_data')
        if name not in data:
            raise AttributeError(f"'{type(self).__name__}' object has no attribute '{name}'")
        
        value = data[name]
        # Recursively wrap nested dicts (with caching)
        if isinstance(value, dict):
            cache = object.__getattribute__(self, '_cache')
            if name not in cache:
                cache[name] = DotDict(value)
            return cache[name]
        return value
    
    def __getitem__(self, key: str) -> Any:
        return self._data[key]
    
    def __contains__(self, key: str) -> bool:
        return key in self._data
    
    def __repr__(self) -> str:
        return f"DotDict({self._data})"
    
    def keys(self):
        return self._data.keys()
    
    def values(self):
        return self._data.values()
    
    def items(self):
        return self._data.items()

logger = structlog.get_logger(__name__)


class VariableStore:
    """
    Variable store with TestStand-style scopes.
    
    Usage:
        store = VariableStore()
        
        # Define variables
        store.define_variable(VariableScope.LOCALS, VariableDef(
            name="voltage", type=PropertyType.NUMBER, default=0.0
        ))
        
        # Set/get values
        store.set("Locals.voltage", 12.5)
        value = store.get("Locals.voltage")  # 12.5
        
        # Nested access
        store.set("Locals.calibration.offset", 0.05)
        offset = store.get("Locals.calibration.offset")
        
        # Array operations
        store.set("Locals.readings[]", 12.0)  # Append
        store.set("Locals.readings[0]", 12.1)  # Update index
        first = store.get("Locals.readings[0]")
    """
    
    def __init__(self, station_file: Optional[Path] = None):
        """
        Initialize variable store.
        
        Args:
            station_file: Path to station globals JSON file for persistence
        """
        self._station_file = station_file
        
        # Variable definitions per scope
        self._definitions: Dict[VariableScope, Dict[str, VariableDef]] = {
            scope: {} for scope in VariableScope
        }
        
        # Container type definitions
        self._container_types: Dict[str, ContainerDef] = {}
        
        # Actual variable values per scope
        self._values: Dict[VariableScope, Dict[str, Any]] = {
            scope: {} for scope in VariableScope
        }
        
        # RunState is read-only, managed by engine
        self._runstate_readonly = True
        
        # Namespace cache for expression evaluation
        self._namespace_cache: Optional[Dict[str, DotDict]] = None
        self._namespace_dirty = True  # Invalidate on any value change
        
        # Load station globals if file exists
        if station_file:
            self._load_station_globals()
    
    # ============================================
    # TYPE DEFINITIONS
    # ============================================
    
    def define_container_type(self, container_def: ContainerDef) -> None:
        """Register a container type definition."""
        self._container_types[container_def.name] = container_def
        logger.debug("Defined container type", name=container_def.name)
    
    def get_container_type(self, name: str) -> Optional[ContainerDef]:
        """Get container type definition by name."""
        return self._container_types.get(name)
    
    def define_variable(
        self, 
        scope: VariableScope, 
        variable_def: VariableDef
    ) -> None:
        """
        Define a variable in a scope.
        
        Initializes the variable with its default value.
        """
        self._definitions[scope][variable_def.name] = variable_def
        
        # Initialize with default value
        self._values[scope][variable_def.name] = variable_def.get_default_value()
        
        logger.debug("Defined variable", 
                    scope=scope.value, 
                    name=variable_def.name,
                    type=variable_def.type.value)
    
    def define_variables_from_dict(
        self, 
        scope: VariableScope, 
        variables: Dict[str, Dict[str, Any]]
    ) -> None:
        """
        Define multiple variables from a dictionary.
        
        Format: {"var_name": {"type": "number", "default": 0.0}}
        """
        for name, var_dict in variables.items():
            var_def = VariableDef(
                name=name,
                type=PropertyType(var_dict.get("type", "string")),
                default=var_dict.get("default"),
                container_type=var_dict.get("container_type"),
                description=var_dict.get("description"),
            )
            self.define_variable(scope, var_def)
    
    # ============================================
    # GET / SET VALUES
    # ============================================
    
    def get(self, reference: str, default: Any = None) -> Any:
        """
        Get variable value using dot-notation reference.
        
        Args:
            reference: Variable reference (e.g., "Locals.voltage")
            default: Default value if not found
            
        Returns:
            Variable value or default
            
        Examples:
            store.get("Locals.voltage")
            store.get("Locals.calibration.offset")
            store.get("Locals.readings[0]")
            store.get("RunState.LoopIndex")
        """
        try:
            ref = VariableReference.parse(reference)
        except ValueError as e:
            logger.warning("Invalid reference", reference=reference, error=str(e))
            return default
        
        # Get scope values
        scope_values = self._values.get(ref.scope, {})
        
        # Navigate path
        value = scope_values
        for i, part in enumerate(ref.path):
            if isinstance(value, dict) and part in value:
                value = value[part]
            else:
                return default
        
        # Handle array index
        if ref.index is not None:
            if isinstance(value, list) and 0 <= ref.index < len(value):
                return value[ref.index]
            return default
        
        return value
    
    def set(self, reference: str, value: Any) -> bool:
        """
        Set variable value using dot-notation reference.
        
        Args:
            reference: Variable reference (e.g., "Locals.voltage")
            value: Value to set
            
        Returns:
            True if successful, False otherwise
            
        Examples:
            store.set("Locals.voltage", 12.5)
            store.set("Locals.calibration.offset", 0.05)
            store.set("Locals.readings[]", 12.0)  # Append
            store.set("Locals.readings[0]", 12.1)  # Update
        """
        try:
            ref = VariableReference.parse(reference)
        except ValueError as e:
            logger.warning("Invalid reference", reference=reference, error=str(e))
            return False
        
        # Check read-only scopes
        if ref.scope == VariableScope.RUNSTATE and self._runstate_readonly:
            logger.warning("Cannot modify RunState from sequence")
            return False
        
        if ref.scope == VariableScope.STEP:
            logger.warning("Cannot modify Step properties from sequence")
            return False
        
        # Get scope values
        scope_values = self._values.get(ref.scope)
        if scope_values is None:
            return False
        
        # Navigate to parent
        parent = scope_values
        for i, part in enumerate(ref.path[:-1]):
            if isinstance(parent, dict):
                if part not in parent:
                    parent[part] = {}
                parent = parent[part]
            else:
                return False
        
        # Get final key
        final_key = ref.path[-1] if ref.path else None
        if final_key is None:
            return False
        
        # Handle array operations
        if ref.append:
            # Append to array
            if final_key not in parent:
                parent[final_key] = []
            if isinstance(parent[final_key], list):
                parent[final_key].append(value)
                logger.debug("Appended to array", reference=reference, value=value)
            else:
                return False
        elif ref.index is not None:
            # Update array index
            if final_key not in parent:
                parent[final_key] = []
            arr = parent[final_key]
            if isinstance(arr, list):
                # Extend if needed
                while len(arr) <= ref.index:
                    arr.append(None)
                arr[ref.index] = value
                logger.debug("Updated array index", reference=reference, index=ref.index, value=value)
            else:
                return False
        else:
            # Simple set
            parent[final_key] = value
            logger.debug("Set variable", reference=reference, value=value)
        
        # Invalidate namespace cache
        self._namespace_dirty = True
        
        # Persist station globals
        if ref.scope == VariableScope.STATION:
            self._save_station_globals()
        
        return True
    
    def exists(self, reference: str) -> bool:
        """Check if a variable reference exists."""
        return self.get(reference, _MISSING) is not _MISSING
    
    # ============================================
    # RUNSTATE MANAGEMENT (called by engine)
    # ============================================
    
    def set_runstate(self, key: str, value: Any) -> None:
        """Set RunState value (engine only)."""
        self._values[VariableScope.RUNSTATE][key] = value
        self._namespace_dirty = True
    
    def update_runstate(self, values: Dict[str, Any]) -> None:
        """Update multiple RunState values."""
        self._values[VariableScope.RUNSTATE].update(values)
        self._namespace_dirty = True
    
    def set_step_result(self, step_id: str, result: Dict[str, Any]) -> None:
        """Set current step result (engine only)."""
        self._values[VariableScope.STEP] = {
            "Id": step_id,
            "Result": result,
        }
        self._namespace_dirty = True
    
    # ============================================
    # SCOPE OPERATIONS
    # ============================================
    
    def get_scope(self, scope: VariableScope) -> Dict[str, Any]:
        """Get all variables in a scope."""
        return self._values.get(scope, {}).copy()
    
    def clear_scope(self, scope: VariableScope) -> None:
        """Clear all variables in a scope (except definitions)."""
        if scope in [VariableScope.RUNSTATE, VariableScope.STEP]:
            self._values[scope] = {}
        else:
            # Reinitialize with defaults
            for name, var_def in self._definitions[scope].items():
                self._values[scope][name] = var_def.get_default_value()
    
    def reset_for_new_execution(self) -> None:
        """Reset variables for a new sequence execution."""
        self.clear_scope(VariableScope.LOCALS)
        self.clear_scope(VariableScope.PARAMETERS)
        self.clear_scope(VariableScope.RUNSTATE)
        self.clear_scope(VariableScope.STEP)
        # Station is preserved
    
    # ============================================
    # PERSISTENCE
    # ============================================
    
    def _load_station_globals(self) -> None:
        """Load station globals from JSON file."""
        if not self._station_file or not self._station_file.exists():
            return
        
        try:
            with open(self._station_file, "r") as f:
                data = json.load(f)
            
            self._values[VariableScope.STATION] = data.get("values", {})
            
            # Load definitions
            for name, var_dict in data.get("definitions", {}).items():
                var_def = VariableDef(
                    name=name,
                    type=PropertyType(var_dict.get("type", "string")),
                    default=var_dict.get("default"),
                    container_type=var_dict.get("container_type"),
                )
                self._definitions[VariableScope.STATION][name] = var_def
            
            logger.info("Loaded station globals", 
                       count=len(self._values[VariableScope.STATION]))
        except Exception as e:
            logger.error("Failed to load station globals", error=str(e))
    
    def _save_station_globals(self) -> None:
        """Save station globals to JSON file."""
        if not self._station_file:
            return
        
        try:
            # Ensure directory exists
            self._station_file.parent.mkdir(parents=True, exist_ok=True)
            
            data = {
                "values": self._values[VariableScope.STATION],
                "definitions": {
                    name: {
                        "type": var_def.type.value,
                        "default": var_def.default,
                        "container_type": var_def.container_type,
                    }
                    for name, var_def in self._definitions[VariableScope.STATION].items()
                }
            }
            
            with open(self._station_file, "w") as f:
                json.dump(data, f, indent=2)
            
            logger.debug("Saved station globals")
        except Exception as e:
            logger.error("Failed to save station globals", error=str(e))
    
    # ============================================
    # NAMESPACE FOR EXPRESSIONS
    # ============================================
    
    def get_expression_namespace(self) -> Dict[str, Any]:
        """
        Get namespace for expression evaluation.
        
        Returns a dict where each scope is accessible as a DotDict,
        enabling dot-notation access like `Locals.voltage`.
        
        Uses caching - only rebuilds when values have changed.
        """
        if self._namespace_dirty or self._namespace_cache is None:
            self._namespace_cache = {
                scope.value: DotDict(values)
                for scope, values in self._values.items()
            }
            self._namespace_dirty = False
        
        return self._namespace_cache


# Sentinel for missing values
class _Missing:
    pass

_MISSING = _Missing()
