"""
SeqMaster Runtime - Variable Types
Layer: EXECUTOR

Type definitions for the variable system, inspired by NI TestStand.

Supports:
- Primitive types: number, string, boolean
- Array types: number[], string[], boolean[]
- Container types: user-defined structured types
- Container arrays: array of containers
"""

from enum import Enum
from typing import Any, Dict, List, Optional, Union

from pydantic import BaseModel, Field


class PropertyType(str, Enum):
    """
    Variable property types - simple Python/Exec compatible types.
    
    - NUMBER: Double precision floating point (also used for integers)
    - STRING: Text string
    - BOOLEAN: True/False
    - OBJECT: Dynamic dict/JSON object (Python dict, Exec JSON)
    - Arrays: Ordered collection of primitives
    """
    NUMBER = "number"
    STRING = "string"
    BOOLEAN = "boolean"
    OBJECT = "object"
    NUMBER_ARRAY = "number[]"
    STRING_ARRAY = "string[]"
    BOOLEAN_ARRAY = "boolean[]"
    
    @classmethod
    def is_array(cls, prop_type: "PropertyType") -> bool:
        """Check if type is an array type."""
        return prop_type in [
            cls.NUMBER_ARRAY,
            cls.STRING_ARRAY,
            cls.BOOLEAN_ARRAY,
        ]
    
    @classmethod
    def get_element_type(cls, array_type: "PropertyType") -> "PropertyType":
        """Get element type for an array type."""
        mapping = {
            cls.NUMBER_ARRAY: cls.NUMBER,
            cls.STRING_ARRAY: cls.STRING,
            cls.BOOLEAN_ARRAY: cls.BOOLEAN,
        }
        return mapping.get(array_type, cls.STRING)
    
    @classmethod
    def python_type(cls, prop_type: "PropertyType") -> type:
        """Get corresponding Python type."""
        mapping = {
            cls.NUMBER: float,
            cls.STRING: str,
            cls.BOOLEAN: bool,
            cls.OBJECT: dict,
            cls.NUMBER_ARRAY: list,
            cls.STRING_ARRAY: list,
            cls.BOOLEAN_ARRAY: list,
        }
        return mapping.get(prop_type, object)


class VariableScope(str, Enum):
    """
    Variable scopes (inspired by TestStand).
    
    Scopes determine variable lifetime and visibility:
    - LOCALS: Variables local to one sequence execution
    - PARAMETERS: Input/output parameters to a sequence (like function args)
    - STATION: Persistent variables shared across all sequences on this tester
    - RUNSTATE: Automatic runtime state (loop index, step info, etc.) - read-only
    - STEP: Current step's result and properties
    - SCANNED: Variables extracted from barcode scanner (operator scans product label)
    """
    LOCALS = "Locals"
    PARAMETERS = "Parameters"
    STATION = "Station"
    RUNSTATE = "RunState"
    STEP = "Step"
    SCANNED = "Scanned"


class VariableDef(BaseModel):
    """
    Variable definition.
    
    Defines a variable's name, type, and default value.
    For container types, references a ContainerDef by name.
    
    Example YAML:
        serial: { type: string }
        voltage: { type: number, default: 0.0 }
        calibration: { type: container, container_type: CalibrationData }
        readings: { type: number[], default: [] }
    """
    name: str = Field(..., description="Variable name")
    type: PropertyType = Field(..., description="Variable type")
    default: Optional[Any] = Field(None, description="Default value")
    container_type: Optional[str] = Field(
        None, 
        description="Container type name (required for container types)"
    )
    description: Optional[str] = Field(None, description="Human-readable description")
    
    def get_default_value(self) -> Any:
        """Get default value based on type."""
        if self.default is not None:
            return self.default
        
        defaults = {
            PropertyType.NUMBER: 0.0,
            PropertyType.STRING: "",
            PropertyType.BOOLEAN: False,
            PropertyType.OBJECT: {},
            PropertyType.NUMBER_ARRAY: [],
            PropertyType.STRING_ARRAY: [],
            PropertyType.BOOLEAN_ARRAY: [],
        }
        return defaults.get(self.type, None)
    
    def validate_value(self, value: Any) -> bool:
        """Check if value matches expected type."""
        if value is None:
            return True  # None is always valid
        
        if PropertyType.is_array(self.type):
            if not isinstance(value, list):
                return False
            element_type = PropertyType.get_element_type(self.type)
            return all(self._validate_element(v, element_type) for v in value)
        
        return self._validate_element(value, self.type)
    
    def _validate_element(self, value: Any, prop_type: PropertyType) -> bool:
        """Validate a single element."""
        if prop_type == PropertyType.NUMBER:
            return isinstance(value, (int, float))
        elif prop_type == PropertyType.STRING:
            return isinstance(value, str)
        elif prop_type == PropertyType.BOOLEAN:
            return isinstance(value, bool)
        elif prop_type == PropertyType.OBJECT:
            return isinstance(value, dict)
        return True


# ContainerDef kept for backwards compatibility but not used in new sequences
class ContainerDef(BaseModel):
    """
    Container (custom type) definition - DEPRECATED.
    
    Kept for backwards compatibility. New sequences should use 'object' type instead.
    
    Example YAML:
        CalibrationData:
            offset: { type: number, default: 0.0 }
            gain: { type: number, default: 1.0 }
            date: { type: string }
    """
    name: str = Field(..., description="Container type name")
    properties: Dict[str, VariableDef] = Field(
        default_factory=dict,
        description="Named properties of the container"
    )
    description: Optional[str] = Field(None, description="Human-readable description")
    
    def create_default_instance(self) -> Dict[str, Any]:
        """Create a new instance with default values."""
        return {
            name: prop.get_default_value()
            for name, prop in self.properties.items()
        }
    
    def validate_instance(self, instance: Dict[str, Any]) -> List[str]:
        """
        Validate an instance against this container definition.
        Returns list of validation errors (empty if valid).
        """
        errors = []
        
        if not isinstance(instance, dict):
            return ["Instance must be a dictionary"]
        
        for name, prop in self.properties.items():
            if name in instance:
                if not prop.validate_value(instance[name]):
                    errors.append(
                        f"Property '{name}': expected {prop.type.value}, "
                        f"got {type(instance[name]).__name__}"
                    )
        
        return errors


class VariableReference(BaseModel):
    """
    Parsed variable reference.
    
    Parses expressions like:
    - "Locals.voltage" -> scope=LOCALS, path=["voltage"]
    - "Locals.calibration.offset" -> scope=LOCALS, path=["calibration", "offset"]
    - "Locals.readings[0]" -> scope=LOCALS, path=["readings"], index=0
    - "Locals.readings[]" -> scope=LOCALS, path=["readings"], append=True
    """
    scope: VariableScope
    path: List[str] = Field(default_factory=list)
    index: Optional[int] = None
    append: bool = False
    
    @classmethod
    def parse(cls, reference: str) -> "VariableReference":
        """
        Parse a variable reference string.
        
        Format: Scope.property.subproperty[index]
        """
        import re
        
        # Extract scope
        parts = reference.split(".", 1)
        if len(parts) < 2:
            raise ValueError(f"Invalid reference: {reference}")
        
        scope_str = parts[0]
        try:
            scope = VariableScope(scope_str)
        except ValueError:
            raise ValueError(f"Invalid scope: {scope_str}")
        
        # Parse path and optional index
        path_str = parts[1]
        path = []
        index = None
        append = False
        
        # Check for array append syntax: "readings[]"
        if path_str.endswith("[]"):
            append = True
            path_str = path_str[:-2]
        
        # Check for array index: "readings[0]"
        index_match = re.search(r'\[(\d+)\]$', path_str)
        if index_match:
            index = int(index_match.group(1))
            path_str = path_str[:index_match.start()]
        
        # Split remaining path
        path = path_str.split(".") if path_str else []
        
        return cls(
            scope=scope,
            path=path,
            index=index,
            append=append
        )
    
    def to_string(self) -> str:
        """Convert back to string representation."""
        result = f"{self.scope.value}.{'.'.join(self.path)}"
        if self.append:
            result += "[]"
        elif self.index is not None:
            result += f"[{self.index}]"
        return result
