"""
SeqMaster Runtime - Hardware Detector
Layer: RUNTIME/HARDWARE

Auto-detects connected hardware including:
- USB devices (barcode scanner, lab PSU, oscilloscope, LabJack, USB-CAN, relay outputs)
- Serial via USB-to-Serial converter
- BLE / Bluetooth devices

Builds a capability tree for the tester.
"""

import asyncio
import re
from dataclasses import dataclass, field
from typing import Dict, List, Optional, Any
from enum import Enum

import structlog

logger = structlog.get_logger(__name__)


class DeviceType(str, Enum):
    """Supported device types."""
    SCPI_PSU = "scpi_psu"
    SCPI_OSCILLOSCOPE = "scpi_oscilloscope"
    SCPI_DMM = "scpi_dmm"
    LABJACK = "labjack"
    CAN_ADAPTER = "can_adapter"
    BARCODE_SCANNER = "barcode_scanner"
    RELAY_OUTPUT = "relay_output"
    SERIAL_DEVICE = "serial_device"
    BLE_DEVICE = "ble_device"
    UNKNOWN = "unknown"


@dataclass
class DetectedDevice:
    """Represents a detected hardware device."""
    device_type: DeviceType
    device_id: str
    name: str
    port: Optional[str] = None
    serial_number: Optional[str] = None
    manufacturer: Optional[str] = None
    properties: Dict[str, Any] = field(default_factory=dict)
    available: bool = True


@dataclass
class CapabilityTree:
    """
    Capability tree representing all available hardware on this tester.
    
    Example:
    {
        "can": true,
        "lab_psu": ["SCPI"],
        "barcode_scanner": true,
        "relay_outputs": 8,
        "oscilloscope": true,
        "ble": true
    }
    """
    devices: List[DetectedDevice] = field(default_factory=list)
    
    # Capability flags
    can: bool = False
    lab_psu: List[str] = field(default_factory=list)
    oscilloscope: bool = False
    dmm: bool = False
    barcode_scanner: bool = False
    relay_outputs: int = 0
    labjack: bool = False
    ble: bool = False
    serial_ports: List[str] = field(default_factory=list)
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary for JSON serialization."""
        return {
            "can": self.can,
            "lab_psu": self.lab_psu,
            "oscilloscope": self.oscilloscope,
            "dmm": self.dmm,
            "barcode_scanner": self.barcode_scanner,
            "relay_outputs": self.relay_outputs,
            "labjack": self.labjack,
            "ble": self.ble,
            "serial_ports": self.serial_ports,
            "devices": [
                {
                    "type": d.device_type.value,
                    "id": d.device_id,
                    "name": d.name,
                    "port": d.port,
                    "serial_number": d.serial_number,
                    "manufacturer": d.manufacturer,
                    "available": d.available
                }
                for d in self.devices
            ]
        }


class HardwareDetector:
    """
    Hardware auto-detection service.
    
    Scans for:
    - USB devices
    - Serial ports
    - VISA instruments (SCPI)
    - LabJack devices
    - BLE devices
    """
    
    def __init__(self):
        self._capabilities = CapabilityTree()
        self._detection_lock = asyncio.Lock()
    
    async def detect_all(self) -> CapabilityTree:
        """
        Run full hardware detection scan.
        
        Returns:
            CapabilityTree with all detected capabilities
        """
        async with self._detection_lock:
            self._capabilities = CapabilityTree()
            
            # Run all detections in parallel where possible
            await asyncio.gather(
                self._detect_usb_devices(),
                self._detect_serial_ports(),
                self._detect_visa_instruments(),
                self._detect_labjack(),
                self._detect_ble_devices(),
                return_exceptions=True
            )
            
            logger.info("Hardware detection complete", 
                       capabilities=self._capabilities.to_dict())
            
            return self._capabilities
    
    async def _detect_usb_devices(self) -> None:
        """Detect USB devices."""
        try:
            import usb.core
            import usb.util
            
            devices = usb.core.find(find_all=True)
            
            for device in devices:
                try:
                    device_info = self._identify_usb_device(device)
                    if device_info:
                        self._capabilities.devices.append(device_info)
                        self._update_capabilities(device_info)
                except Exception as e:
                    logger.debug("Could not identify USB device", error=str(e))
                    
        except ImportError:
            logger.warning("pyusb not available, USB detection disabled")
        except Exception as e:
            logger.error("USB detection failed", error=str(e))
    
    def _identify_usb_device(self, device) -> Optional[DetectedDevice]:
        """Identify a USB device by VID/PID."""
        vid = device.idVendor
        pid = device.idProduct
        
        # Known device mappings (expand as needed)
        known_devices = {
            # LabJack
            (0x0CD5, 0x0003): (DeviceType.LABJACK, "LabJack U3"),
            (0x0CD5, 0x0009): (DeviceType.LABJACK, "LabJack U6"),
            (0x0CD5, 0x000C): (DeviceType.LABJACK, "LabJack T7"),
            # CANable / USB-CAN adapters
            (0x1D50, 0x606F): (DeviceType.CAN_ADAPTER, "CANable"),
            (0x0483, 0x5740): (DeviceType.CAN_ADAPTER, "STM32 CAN"),
            # Barcode scanners (common VIDs)
            (0x05E0, None): (DeviceType.BARCODE_SCANNER, "Barcode Scanner"),
            (0x0C2E, None): (DeviceType.BARCODE_SCANNER, "Honeywell Scanner"),
        }
        
        # Check exact match first
        if (vid, pid) in known_devices:
            device_type, name = known_devices[(vid, pid)]
            return DetectedDevice(
                device_type=device_type,
                device_id=f"USB-{vid:04X}:{pid:04X}",
                name=name,
                manufacturer=str(device.manufacturer) if device.manufacturer else None,
                serial_number=str(device.serial_number) if device.serial_number else None
            )
        
        # Check VID-only match
        for (known_vid, known_pid), (device_type, name) in known_devices.items():
            if known_vid == vid and known_pid is None:
                return DetectedDevice(
                    device_type=device_type,
                    device_id=f"USB-{vid:04X}:{pid:04X}",
                    name=name,
                    manufacturer=str(device.manufacturer) if device.manufacturer else None
                )
        
        return None
    
    async def _detect_serial_ports(self) -> None:
        """Detect serial ports."""
        try:
            import serial.tools.list_ports
            
            ports = serial.tools.list_ports.comports()
            
            for port in ports:
                self._capabilities.serial_ports.append(port.device)
                
                device = DetectedDevice(
                    device_type=DeviceType.SERIAL_DEVICE,
                    device_id=port.device,
                    name=port.description or "Serial Port",
                    port=port.device,
                    manufacturer=port.manufacturer,
                    serial_number=port.serial_number
                )
                self._capabilities.devices.append(device)
                
                # Try to identify device type from description
                if port.description:
                    desc_lower = port.description.lower()
                    if "can" in desc_lower:
                        self._capabilities.can = True
                    elif "relay" in desc_lower:
                        self._capabilities.relay_outputs += 8  # Assume 8-channel
                        
        except ImportError:
            logger.warning("pyserial not available, serial detection disabled")
        except Exception as e:
            logger.error("Serial port detection failed", error=str(e))
    
    async def _detect_visa_instruments(self) -> None:
        """Detect VISA/SCPI instruments."""
        try:
            import pyvisa
            
            rm = pyvisa.ResourceManager('@py')
            resources = rm.list_resources()
            
            for resource in resources:
                try:
                    inst = rm.open_resource(resource, timeout=2000)
                    idn = inst.query("*IDN?").strip()
                    inst.close()
                    
                    device_type, name = self._identify_scpi_device(idn)
                    
                    device = DetectedDevice(
                        device_type=device_type,
                        device_id=resource,
                        name=name,
                        port=resource,
                        properties={"idn": idn}
                    )
                    self._capabilities.devices.append(device)
                    self._update_capabilities(device)
                    
                except Exception as e:
                    logger.debug("Could not query VISA resource", 
                               resource=resource, error=str(e))
                    
        except ImportError:
            logger.warning("pyvisa not available, SCPI detection disabled")
        except Exception as e:
            logger.error("VISA detection failed", error=str(e))
    
    def _identify_scpi_device(self, idn: str) -> tuple:
        """Identify SCPI device from *IDN? response."""
        idn_lower = idn.lower()
        
        if any(x in idn_lower for x in ["psu", "power supply", "e36", "n57", "n67"]):
            return DeviceType.SCPI_PSU, f"SCPI PSU: {idn}"
        elif any(x in idn_lower for x in ["oscilloscope", "dso", "mso", "scope"]):
            return DeviceType.SCPI_OSCILLOSCOPE, f"Oscilloscope: {idn}"
        elif any(x in idn_lower for x in ["dmm", "multimeter", "34"]):
            return DeviceType.SCPI_DMM, f"DMM: {idn}"
        else:
            return DeviceType.UNKNOWN, f"SCPI Device: {idn}"
    
    async def _detect_labjack(self) -> None:
        """Detect LabJack devices."""
        try:
            # Try LabJack LJM library
            from labjack import ljm
            
            device_types = [ljm.constants.dtT7, ljm.constants.dtT4, ljm.constants.dtDIGIT]
            
            for dt in device_types:
                try:
                    handle = ljm.openS(dt, "ANY", "ANY")
                    info = ljm.getHandleInfo(handle)
                    ljm.close(handle)
                    
                    device = DetectedDevice(
                        device_type=DeviceType.LABJACK,
                        device_id=f"LJ-{info[2]}",
                        name=f"LabJack {['T7', 'T4', 'DIGIT'][device_types.index(dt)]}",
                        serial_number=str(info[2])
                    )
                    self._capabilities.devices.append(device)
                    self._capabilities.labjack = True
                    
                except ljm.LJMError:
                    pass
                    
        except ImportError:
            logger.debug("LabJack LJM not available")
        except Exception as e:
            logger.error("LabJack detection failed", error=str(e))
    
    async def _detect_ble_devices(self) -> None:
        """Detect BLE devices."""
        try:
            from bleak import BleakScanner
            from src.core.config import settings
            
            devices = await BleakScanner.discover(
                timeout=settings.BLE_SCAN_TIMEOUT_SECONDS
            )
            
            if devices:
                self._capabilities.ble = True
                
            for device in devices:
                # Handle different bleak versions - rssi may be on device or in advertisement_data
                rssi = getattr(device, 'rssi', None)
                ble_device = DetectedDevice(
                    device_type=DeviceType.BLE_DEVICE,
                    device_id=device.address,
                    name=device.name or "Unknown BLE Device",
                    properties={"rssi": rssi} if rssi is not None else {}
                )
                self._capabilities.devices.append(ble_device)
                
        except ImportError:
            logger.debug("bleak not available, BLE detection disabled")
        except Exception as e:
            logger.error("BLE detection failed", error=str(e))
    
    def _update_capabilities(self, device: DetectedDevice) -> None:
        """Update capability flags based on detected device."""
        if device.device_type == DeviceType.SCPI_PSU:
            if "SCPI" not in self._capabilities.lab_psu:
                self._capabilities.lab_psu.append("SCPI")
        elif device.device_type == DeviceType.SCPI_OSCILLOSCOPE:
            self._capabilities.oscilloscope = True
        elif device.device_type == DeviceType.SCPI_DMM:
            self._capabilities.dmm = True
        elif device.device_type == DeviceType.LABJACK:
            self._capabilities.labjack = True
        elif device.device_type == DeviceType.CAN_ADAPTER:
            self._capabilities.can = True
        elif device.device_type == DeviceType.BARCODE_SCANNER:
            self._capabilities.barcode_scanner = True
        elif device.device_type == DeviceType.RELAY_OUTPUT:
            self._capabilities.relay_outputs += 8
        elif device.device_type == DeviceType.BLE_DEVICE:
            self._capabilities.ble = True
    
    @property
    def capabilities(self) -> CapabilityTree:
        """Get current capabilities."""
        return self._capabilities
    
    async def refresh(self) -> CapabilityTree:
        """Refresh hardware detection."""
        return await self.detect_all()
