"""
SeqMaster Runtime - Main Entry Point
Layer: RUNTIME

This is the main entry point for the SeqMaster runtime service
running on Raspberry Pi 5 as an edge node.
"""

import asyncio
import signal
import sys
from contextlib import asynccontextmanager

import structlog
import uvicorn
from fastapi import FastAPI
from fastapi.middleware.cors import CORSMiddleware

from src.api.routes import api_router
from src.api.auth import auth_router
from src.api.websocket import websocket_router
from src.api.network import router as network_router
from src.api.analytics import router as analytics_router
from src.api.license import license_router
from src.core.config import settings
from src.core.watchdog import WatchdogService
from src.database.connection import init_database, close_database
from src.hardware.detector import HardwareDetector
from src.core.service_registry import ServiceRegistry

# Configure structured logging
structlog.configure(
    processors=[
        structlog.stdlib.filter_by_level,
        structlog.stdlib.add_logger_name,
        structlog.stdlib.add_log_level,
        structlog.stdlib.PositionalArgumentsFormatter(),
        structlog.processors.TimeStamper(fmt="iso"),
        structlog.processors.StackInfoRenderer(),
        structlog.processors.format_exc_info,
        structlog.processors.UnicodeDecoder(),
        structlog.processors.JSONRenderer()
    ],
    wrapper_class=structlog.stdlib.BoundLogger,
    context_class=dict,
    logger_factory=structlog.stdlib.LoggerFactory(),
    cache_logger_on_first_use=True,
)

logger = structlog.get_logger(__name__)


@asynccontextmanager
async def lifespan(app: FastAPI):
    """Application lifespan manager for startup and shutdown."""
    import time
    logger.info("SeqMaster Runtime starting...", version=settings.VERSION)
    
    # Record start time for uptime tracking
    ServiceRegistry.set("start_time", time.time())
    
    # Initialize database
    await init_database()
    logger.info("Database initialized")
    
    # Check for password reset file (requires physical SD-card access)
    from src.security.setup import check_and_process_reset_file
    from src.database.connection import async_session_maker
    async with async_session_maker() as db:
        reset_processed = await check_and_process_reset_file(db)
        if reset_processed:
            logger.warning("Admin password reset processed via SD-card file")
    
    # Initialize hardware detector
    hardware_detector = HardwareDetector()
    capabilities = await hardware_detector.detect_all()
    ServiceRegistry.set("hardware_detector", hardware_detector)
    ServiceRegistry.set("capabilities", capabilities)
    logger.info("Hardware detection complete", capabilities=capabilities)
    
    # Start watchdog service
    watchdog = WatchdogService()
    await watchdog.start()
    ServiceRegistry.set("watchdog", watchdog)
    logger.info("Watchdog service started")
    
    # Register with mDNS for discovery
    from src.core.discovery import DiscoveryService
    discovery = DiscoveryService()
    await discovery.register()
    ServiceRegistry.set("discovery", discovery)
    logger.info("mDNS discovery registered", 
                hostname=settings.HOSTNAME,
                port=settings.API_PORT)
    
    # Start sync service (Option D: Offline-First Architecture)
    from src.sync.service import SyncService
    sync_service = SyncService()
    await sync_service.start()
    ServiceRegistry.set("sync_service", sync_service)
    logger.info("Sync service started",
                enabled=sync_service.is_enabled,
                central_url=settings.CENTRAL_URL or "not configured")
    
    # Log license status
    from src.core.license import get_license_service, DEMO_LIMITS
    lic = get_license_service()
    lic_info = lic.get_license_info()
    if lic.is_demo():
        logger.warning("Running in DEMO mode",
                       hardware_id=lic_info.hardware_id,
                       limits=DEMO_LIMITS)
    else:
        logger.info("License active",
                     tier=lic_info.tier.value,
                     license_id=lic_info.license_id,
                     customer=lic_info.customer_name,
                     expires_at=str(lic_info.expires_at) if lic_info.expires_at else "never")
    
    yield
    
    # Shutdown
    logger.info("SeqMaster Runtime shutting down...")
    await sync_service.stop()
    await discovery.unregister()
    await watchdog.stop()
    await close_database()
    logger.info("Shutdown complete")


def create_app() -> FastAPI:
    """Create and configure the FastAPI application."""
    app = FastAPI(
        title="SeqMaster Runtime API",
        description="Edge-first test system runtime for Raspberry Pi 5",
        version=settings.VERSION,
        lifespan=lifespan
    )
    
    # CORS middleware for interface access
    app.add_middleware(
        CORSMiddleware,
        allow_origins=["*"],
        allow_credentials=False,
        allow_methods=["*"],
        allow_headers=["*"],
    )
    
    # Include routers
    app.include_router(auth_router, prefix="/api/v1/auth")
    app.include_router(api_router, prefix="/api/v1")
    app.include_router(network_router, prefix="/api/v1")
    app.include_router(analytics_router, prefix="/api/v1")
    app.include_router(license_router, prefix="/api/v1")
    app.include_router(websocket_router, prefix="/ws")
    
    # Root health endpoint (for load balancers, monitoring)
    @app.get("/health")
    async def root_health():
        from datetime import datetime
        capabilities = ServiceRegistry.get("capabilities")
        return {
            "status": "healthy",
            "tester_id": settings.TESTER_ID,
            "version": settings.VERSION,
            "timestamp": datetime.utcnow().isoformat(),
            "hardware_detected": len(capabilities.devices) if capabilities else 0
        }
    
    return app


app = create_app()


def handle_signal(signum, frame):
    """Handle shutdown signals gracefully."""
    logger.info("Received shutdown signal", signal=signum)
    sys.exit(0)


if __name__ == "__main__":
    # Register signal handlers
    signal.signal(signal.SIGINT, handle_signal)
    signal.signal(signal.SIGTERM, handle_signal)
    
    # Run the server
    uvicorn.run(
        "src.main:app",
        host=settings.API_HOST,
        port=settings.API_PORT,
        reload=settings.DEBUG,
        log_level="info"
    )
