"""
SeqMaster Runtime - Authentication & Authorization
Layer: SECURITY

Provides:
- JWT token-based authentication
- Role-based access control
- Password hashing
"""

from datetime import datetime, timedelta
from typing import Optional, Callable
from functools import wraps

from fastapi import Depends, HTTPException, status
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
from jose import JWTError, jwt
from argon2 import PasswordHasher
from argon2.exceptions import VerifyMismatchError
from pydantic import BaseModel

from src.core.config import settings
from src.database.models import User, UserRole
from src.database.connection import get_db_session
from src.database.repository import UserRepository

# Password hashing with Argon2 (more secure than bcrypt)
ph = PasswordHasher()

# Token bearer
security = HTTPBearer(auto_error=False)


class TokenData(BaseModel):
    """Token payload data."""
    username: str
    user_id: int
    role: UserRole
    exp: datetime


class TokenResponse(BaseModel):
    """Token response."""
    access_token: str
    token_type: str = "bearer"
    expires_at: datetime
    user: dict


# ============================================
# PASSWORD UTILITIES
# ============================================

def hash_password(password: str) -> str:
    """Hash a password using Argon2."""
    return ph.hash(password)


def verify_password(plain_password: str, hashed_password: str) -> bool:
    """Verify a password against its hash."""
    try:
        ph.verify(hashed_password, plain_password)
        return True
    except VerifyMismatchError:
        return False


# ============================================
# TOKEN UTILITIES
# ============================================

def create_access_token(user: User) -> tuple[str, datetime]:
    """
    Create a JWT access token.
    
    Returns:
        Tuple of (token, expiry_datetime)
    """
    expires_at = datetime.utcnow() + timedelta(minutes=settings.ACCESS_TOKEN_EXPIRE_MINUTES)
    
    payload = {
        "sub": user.username,
        "user_id": user.id,
        "role": user.role.value,
        "exp": expires_at
    }
    
    token = jwt.encode(payload, settings.SECRET_KEY, algorithm=settings.JWT_ALGORITHM)
    return token, expires_at


def decode_token(token: str) -> TokenData:
    """
    Decode and validate a JWT token.
    
    Raises:
        HTTPException if token is invalid
    """
    try:
        payload = jwt.decode(
            token, 
            settings.SECRET_KEY, 
            algorithms=[settings.JWT_ALGORITHM]
        )
        
        return TokenData(
            username=payload["sub"],
            user_id=payload["user_id"],
            role=UserRole(payload["role"]),
            exp=datetime.fromtimestamp(payload["exp"])
        )
        
    except JWTError as e:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Invalid or expired token",
            headers={"WWW-Authenticate": "Bearer"}
        )


# ============================================
# AUTHENTICATION
# ============================================

async def authenticate_user(username: str, password: str, db) -> Optional[User]:
    """
    Authenticate a user by username and password.
    
    Returns:
        User if authentication successful, None otherwise
    """
    repo = UserRepository(db)
    user = await repo.get_by_username(username)
    
    if not user:
        return None
    
    if not user.active:
        return None
    
    if not verify_password(password, user.password_hash):
        return None
    
    # Update last login
    await repo.update_last_login(user.id)
    
    return user


async def get_current_user(
    credentials: Optional[HTTPAuthorizationCredentials] = Depends(security),
    db = Depends(get_db_session)
) -> Optional[User]:
    """
    Get current user from JWT token.
    
    This is a soft dependency - returns None if no token provided.
    Use require_auth for endpoints that require authentication.
    """
    if not credentials:
        return None
    
    token_data = decode_token(credentials.credentials)
    
    repo = UserRepository(db)
    user = await repo.get_by_id(token_data.user_id)
    
    if not user or not user.active:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="User not found or inactive"
        )
    
    return user


async def require_auth(
    credentials: HTTPAuthorizationCredentials = Depends(security),
    db = Depends(get_db_session)
) -> User:
    """
    Require authentication - use as dependency for protected endpoints.
    
    Raises:
        HTTPException if not authenticated
    """
    if not credentials:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Authentication required",
            headers={"WWW-Authenticate": "Bearer"}
        )
    
    user = await get_current_user(credentials, db)
    
    if not user:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Authentication required"
        )
    
    return user


def require_role(required_role: UserRole):
    """
    Create a dependency that requires a specific role or higher.
    
    Role hierarchy: ADMIN > ENGINEER > OPERATOR
    
    Usage:
        @router.get("/admin-only")
        async def admin_endpoint(user = Depends(require_role(UserRole.ADMIN))):
            ...
    """
    role_hierarchy = {
        UserRole.OPERATOR: 0,
        UserRole.ENGINEER: 1,
        UserRole.ADMIN: 2
    }
    
    async def role_checker(
        credentials: HTTPAuthorizationCredentials = Depends(security),
        db = Depends(get_db_session)
    ) -> User:
        if not credentials:
            raise HTTPException(
                status_code=status.HTTP_401_UNAUTHORIZED,
                detail="Authentication required"
            )
        
        user = await get_current_user(credentials, db)
        
        if not user:
            raise HTTPException(
                status_code=status.HTTP_401_UNAUTHORIZED,
                detail="Authentication required"
            )
        
        user_level = role_hierarchy.get(user.role, -1)
        required_level = role_hierarchy.get(required_role, 999)
        
        if user_level < required_level:
            raise HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=f"Requires {required_role.value} role or higher"
            )
        
        return user
    
    return role_checker


def require_permission(permission: str):
    """
    Create a dependency that requires a specific permission.
    
    Checks user's effective permissions (role defaults + custom grants/revokes).
    
    Usage:
        @router.delete("/results/{id}")
        async def delete_result(user = Depends(require_permission("delete_results"))):
            ...
    """
    from src.security.permissions import Permission, get_default_permissions
    
    async def permission_checker(
        credentials: HTTPAuthorizationCredentials = Depends(security),
        db = Depends(get_db_session)
    ) -> User:
        if not credentials:
            raise HTTPException(
                status_code=status.HTTP_401_UNAUTHORIZED,
                detail="Authentication required",
                headers={"WWW-Authenticate": "Bearer"}
            )
        
        user = await get_current_user(credentials, db)
        
        if not user:
            raise HTTPException(
                status_code=status.HTTP_401_UNAUTHORIZED,
                detail="Authentication required"
            )
        
        # Get user's effective permissions (includes db query for custom permissions)
        effective_permissions = await get_user_permissions(user, db)
        
        if permission not in effective_permissions:
            raise HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=f"Permission denied: requires '{permission}'"
            )
        
        return user
    
    return permission_checker


async def get_user_permissions(user: User, db) -> set[str]:
    """
    Get effective permissions for a user.
    
    Combines role defaults with any custom permission grants/revokes.
    Fetches custom permissions from database.
    """
    from src.security.permissions import get_default_permissions
    from src.database.models import UserPermission
    from sqlalchemy import select
    
    # Start with role defaults
    permissions = set(get_default_permissions(user.role))
    
    # Query custom permission overrides from DB
    result = await db.execute(
        select(UserPermission).where(UserPermission.user_id == user.id)
    )
    custom_permissions = result.scalars().all()
    
    # Apply custom permission overrides
    for perm in custom_permissions:
        if perm.granted:
            permissions.add(perm.permission)
        else:
            permissions.discard(perm.permission)
    
    return permissions


# ============================================
# WEBSOCKET AUTHENTICATION
# ============================================

async def verify_token_ws(token: str) -> TokenData:
    """
    Verify a token for WebSocket connections.
    
    Returns:
        TokenData if valid
        
    Raises:
        Exception if invalid
    """
    return decode_token(token)


# ============================================
# USER MANAGEMENT
# ============================================

async def create_user(
    username: str,
    password: str,
    role: UserRole,
    db,
    full_name: Optional[str] = None,
    email: Optional[str] = None
) -> User:
    """Create a new user."""
    repo = UserRepository(db)
    
    # Check if username exists
    existing = await repo.get_by_username(username)
    if existing:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Username already exists"
        )
    
    password_hash = hash_password(password)
    
    user = await repo.create(
        username=username,
        password_hash=password_hash,
        role=role,
        full_name=full_name,
        email=email
    )
    
    return user


# ============================================
# INITIAL ADMIN SETUP
# ============================================

async def ensure_admin_exists(db) -> None:
    """
    Ensure at least one admin user exists.
    Creates default admin if none exist.
    """
    repo = UserRepository(db)
    
    # Check for any admin
    users = await repo.get_all_active()
    admins = [u for u in users if u.role == UserRole.ADMIN]
    
    if not admins:
        # Create default admin
        await repo.create(
            username="admin",
            password_hash=hash_password("admin"),  # Change in production!
            role=UserRole.ADMIN,
            full_name="System Administrator"
        )
