"""
SeqMaster Runtime - Permission System
Layer: SECURITY

Defines all permissions and default role-based access.
Admin can override permissions per user.
"""

from enum import Enum
from typing import Dict, List, Set, Optional
from dataclasses import dataclass

from src.database.models import UserRole


class Permission(str, Enum):
    """All available permissions in the system."""
    
    # Test execution
    RUN_TESTS = "run_tests"
    ABORT_TESTS = "abort_tests"
    
    # Results
    VIEW_RESULTS = "view_results"
    EXPORT_RESULTS = "export_results"
    DELETE_RESULTS = "delete_results"
    
    # Sequences
    VIEW_SEQUENCES = "view_sequences"
    CREATE_SEQUENCES = "create_sequences"
    EDIT_SEQUENCES = "edit_sequences"
    DELETE_SEQUENCES = "delete_sequences"
    PUBLISH_SEQUENCES = "publish_sequences"
    
    # Property Sets
    VIEW_PROPERTY_SETS = "view_property_sets"
    MANAGE_PROPERTY_SETS = "manage_property_sets"
    
    # Hardware
    VIEW_HARDWARE = "view_hardware"
    CONFIGURE_HARDWARE = "configure_hardware"
    
    # System
    VIEW_SYSTEM_INFO = "view_system_info"
    CONFIGURE_SYSTEM = "configure_system"
    CONFIGURE_NETWORK = "configure_network"
    SHUTDOWN_REBOOT = "shutdown_reboot"
    
    # User management
    VIEW_USERS = "view_users"
    MANAGE_USERS = "manage_users"
    MANAGE_PERMISSIONS = "manage_permissions"


# Default permissions per role
# Higher roles inherit all permissions from lower roles
DEFAULT_PERMISSIONS: Dict[UserRole, Set[Permission]] = {
    UserRole.OPERATOR: {
        Permission.RUN_TESTS,
        Permission.ABORT_TESTS,
        Permission.VIEW_RESULTS,
        Permission.EXPORT_RESULTS,
        Permission.VIEW_SEQUENCES,
        Permission.VIEW_PROPERTY_SETS,
        Permission.VIEW_HARDWARE,
        Permission.VIEW_SYSTEM_INFO,
    },
    
    UserRole.ENGINEER: {
        # Inherits OPERATOR permissions plus:
        Permission.RUN_TESTS,
        Permission.ABORT_TESTS,
        Permission.VIEW_RESULTS,
        Permission.EXPORT_RESULTS,
        Permission.DELETE_RESULTS,
        Permission.VIEW_SEQUENCES,
        Permission.CREATE_SEQUENCES,
        Permission.EDIT_SEQUENCES,
        Permission.PUBLISH_SEQUENCES,
        Permission.VIEW_PROPERTY_SETS,
        Permission.MANAGE_PROPERTY_SETS,
        Permission.VIEW_HARDWARE,
        Permission.CONFIGURE_HARDWARE,
        Permission.VIEW_SYSTEM_INFO,
        Permission.VIEW_USERS,
    },
    
    UserRole.ADMIN: {
        # All permissions
        Permission.RUN_TESTS,
        Permission.ABORT_TESTS,
        Permission.VIEW_RESULTS,
        Permission.EXPORT_RESULTS,
        Permission.DELETE_RESULTS,
        Permission.VIEW_SEQUENCES,
        Permission.CREATE_SEQUENCES,
        Permission.EDIT_SEQUENCES,
        Permission.DELETE_SEQUENCES,
        Permission.PUBLISH_SEQUENCES,
        Permission.VIEW_PROPERTY_SETS,
        Permission.MANAGE_PROPERTY_SETS,
        Permission.VIEW_HARDWARE,
        Permission.CONFIGURE_HARDWARE,
        Permission.VIEW_SYSTEM_INFO,
        Permission.CONFIGURE_SYSTEM,
        Permission.CONFIGURE_NETWORK,
        Permission.SHUTDOWN_REBOOT,
        Permission.VIEW_USERS,
        Permission.MANAGE_USERS,
        Permission.MANAGE_PERMISSIONS,
    },
}


@dataclass
class PermissionInfo:
    """Information about a permission for UI display."""
    permission: Permission
    name: str
    description: str
    category: str


# Permission metadata for UI
PERMISSION_INFO: Dict[Permission, PermissionInfo] = {
    Permission.RUN_TESTS: PermissionInfo(
        Permission.RUN_TESTS, "Run Tests", 
        "Start test executions", "Testing"
    ),
    Permission.ABORT_TESTS: PermissionInfo(
        Permission.ABORT_TESTS, "Abort Tests",
        "Stop running tests", "Testing"
    ),
    Permission.VIEW_RESULTS: PermissionInfo(
        Permission.VIEW_RESULTS, "View Results",
        "View test results and history", "Results"
    ),
    Permission.EXPORT_RESULTS: PermissionInfo(
        Permission.EXPORT_RESULTS, "Export Results",
        "Download test reports", "Results"
    ),
    Permission.DELETE_RESULTS: PermissionInfo(
        Permission.DELETE_RESULTS, "Delete Results",
        "Remove test results from history", "Results"
    ),
    Permission.VIEW_SEQUENCES: PermissionInfo(
        Permission.VIEW_SEQUENCES, "View Sequences",
        "View available test sequences", "Sequences"
    ),
    Permission.CREATE_SEQUENCES: PermissionInfo(
        Permission.CREATE_SEQUENCES, "Create Sequences",
        "Create new test sequences", "Sequences"
    ),
    Permission.EDIT_SEQUENCES: PermissionInfo(
        Permission.EDIT_SEQUENCES, "Edit Sequences",
        "Modify existing test sequences", "Sequences"
    ),
    Permission.DELETE_SEQUENCES: PermissionInfo(
        Permission.DELETE_SEQUENCES, "Delete Sequences",
        "Remove test sequences", "Sequences"
    ),
    Permission.PUBLISH_SEQUENCES: PermissionInfo(
        Permission.PUBLISH_SEQUENCES, "Publish Sequences",
        "Publish sequences for production use", "Sequences"
    ),
    Permission.VIEW_PROPERTY_SETS: PermissionInfo(
        Permission.VIEW_PROPERTY_SETS, "View Property Sets",
        "View limit/property sets", "Property Sets"
    ),
    Permission.MANAGE_PROPERTY_SETS: PermissionInfo(
        Permission.MANAGE_PROPERTY_SETS, "Manage Property Sets",
        "Create and edit property sets", "Property Sets"
    ),
    Permission.VIEW_HARDWARE: PermissionInfo(
        Permission.VIEW_HARDWARE, "View Hardware",
        "View connected hardware status", "Hardware"
    ),
    Permission.CONFIGURE_HARDWARE: PermissionInfo(
        Permission.CONFIGURE_HARDWARE, "Configure Hardware",
        "Configure hardware settings", "Hardware"
    ),
    Permission.VIEW_SYSTEM_INFO: PermissionInfo(
        Permission.VIEW_SYSTEM_INFO, "View System Info",
        "View system information and logs", "System"
    ),
    Permission.CONFIGURE_SYSTEM: PermissionInfo(
        Permission.CONFIGURE_SYSTEM, "Configure System",
        "Change system settings", "System"
    ),
    Permission.CONFIGURE_NETWORK: PermissionInfo(
        Permission.CONFIGURE_NETWORK, "Configure Network",
        "Configure network settings", "System"
    ),
    Permission.SHUTDOWN_REBOOT: PermissionInfo(
        Permission.SHUTDOWN_REBOOT, "Shutdown/Reboot",
        "Shutdown or reboot the system", "System"
    ),
    Permission.VIEW_USERS: PermissionInfo(
        Permission.VIEW_USERS, "View Users",
        "View user list", "Administration"
    ),
    Permission.MANAGE_USERS: PermissionInfo(
        Permission.MANAGE_USERS, "Manage Users",
        "Create, edit, and delete users", "Administration"
    ),
    Permission.MANAGE_PERMISSIONS: PermissionInfo(
        Permission.MANAGE_PERMISSIONS, "Manage Permissions",
        "Change user permissions", "Administration"
    ),
}


def get_all_permissions() -> List[dict]:
    """Get all permissions with metadata for UI."""
    result = []
    for perm in Permission:
        info = PERMISSION_INFO.get(perm)
        if info:
            result.append({
                "permission": perm.value,
                "name": info.name,
                "description": info.description,
                "category": info.category
            })
    return result


def get_default_permissions(role: UserRole) -> Set[Permission]:
    """Get default permissions for a role."""
    return DEFAULT_PERMISSIONS.get(role, set())


def get_permissions_by_category() -> Dict[str, List[dict]]:
    """Get permissions grouped by category."""
    categories: Dict[str, List[dict]] = {}
    
    for perm in Permission:
        info = PERMISSION_INFO.get(perm)
        if info:
            if info.category not in categories:
                categories[info.category] = []
            categories[info.category].append({
                "permission": perm.value,
                "name": info.name,
                "description": info.description
            })
    
    return categories
