"""
SeqMaster Runtime - Setup Service
Layer: SECURITY

Handles:
- Initial system setup (first admin user)
- Password reset via SD-card file
- Setup state management
"""

import os
import json
import structlog
from pathlib import Path
from typing import Optional
from datetime import datetime

from sqlalchemy import select, func
from sqlalchemy.ext.asyncio import AsyncSession

from src.database.models import User, UserRole, SystemConfig
from src.security.auth import hash_password

logger = structlog.get_logger(__name__)

# Location for reset file on SD-card (boot partition is readable on most systems)
RESET_FILE_PATHS = [
    "/boot/seqmaster_reset.json",
    "/boot/firmware/seqmaster_reset.json",  # Newer Raspberry Pi OS
    "/media/boot/seqmaster_reset.json",
]

# Alternative: use a file in the data directory that can be created manually
DATA_RESET_PATH = Path(__file__).parent.parent.parent / "data" / "ADMIN_RESET.json"


async def is_setup_complete(db: AsyncSession) -> bool:
    """Check if initial setup has been completed (any user exists)."""
    result = await db.execute(select(func.count(User.id)))
    user_count = result.scalar()
    return user_count > 0


async def get_setup_state(db: AsyncSession) -> dict:
    """Get current setup state for frontend."""
    setup_complete = await is_setup_complete(db)
    
    return {
        "setup_complete": setup_complete,
        "requires_initial_setup": not setup_complete,
        "tester_id": os.environ.get("VERIMATRIX_TESTER_ID", "TESTER-001"),
    }


async def create_initial_admin(
    db: AsyncSession,
    username: str,
    password: str,
    full_name: Optional[str] = None,
    email: Optional[str] = None
) -> User:
    """
    Create the initial admin user during first-time setup.
    
    This can only be called when no users exist in the database.
    """
    # Verify no users exist
    if await is_setup_complete(db):
        raise ValueError("Setup already complete. Cannot create initial admin.")
    
    # Create admin user
    admin = User(
        username=username,
        password_hash=hash_password(password),
        role=UserRole.ADMIN,
        full_name=full_name or "Administrator",
        email=email,
        active=True
    )
    
    db.add(admin)
    await db.commit()
    await db.refresh(admin)
    
    logger.info("Initial admin user created", username=username)
    
    # Store setup timestamp
    config = SystemConfig(
        key="initial_setup_at",
        value=datetime.utcnow().isoformat()
    )
    db.add(config)
    await db.commit()
    
    return admin


def find_reset_file() -> Optional[Path]:
    """Find password reset file on SD-card or data directory."""
    # Check boot partition paths
    for path_str in RESET_FILE_PATHS:
        path = Path(path_str)
        if path.exists():
            return path
    
    # Check data directory
    if DATA_RESET_PATH.exists():
        return DATA_RESET_PATH
    
    return None


async def check_and_process_reset_file(db: AsyncSession) -> bool:
    """
    Check for password reset file and process it.
    
    Reset file format (JSON):
    {
        "action": "reset_admin",
        "new_password": "new_secure_password",
        "confirm": "I understand this resets the admin password"
    }
    
    Returns True if reset was processed.
    """
    reset_file = find_reset_file()
    
    if not reset_file:
        return False
    
    logger.warning("Password reset file found", path=str(reset_file))
    
    try:
        with open(reset_file, 'r') as f:
            data = json.load(f)
        
        # Validate reset request
        if data.get("action") != "reset_admin":
            logger.error("Invalid reset file: wrong action")
            return False
        
        if data.get("confirm") != "I understand this resets the admin password":
            logger.error("Invalid reset file: confirmation missing")
            return False
        
        new_password = data.get("new_password")
        if not new_password or len(new_password) < 6:
            logger.error("Invalid reset file: password too short (min 6 chars)")
            return False
        
        # Find admin user (or first user if no admin exists)
        result = await db.execute(
            select(User).where(User.role == UserRole.ADMIN).limit(1)
        )
        admin = result.scalar_one_or_none()
        
        if not admin:
            # No admin exists - check if any users exist
            result = await db.execute(select(User).limit(1))
            admin = result.scalar_one_or_none()
        
        if admin:
            # Reset existing admin password
            admin.password_hash = hash_password(new_password)
            admin.active = True  # Re-activate if disabled
            await db.commit()
            logger.warning("Admin password reset via SD-card file", 
                          username=admin.username)
        else:
            # No users exist - create new admin
            admin = User(
                username="admin",
                password_hash=hash_password(new_password),
                role=UserRole.ADMIN,
                full_name="Administrator",
                active=True
            )
            db.add(admin)
            await db.commit()
            logger.warning("Admin user created via SD-card reset file")
        
        # Delete the reset file for security
        try:
            reset_file.unlink()
            logger.info("Reset file deleted after processing")
        except Exception as e:
            logger.error("Could not delete reset file", error=str(e))
            # Try to at least clear the contents
            try:
                with open(reset_file, 'w') as f:
                    f.write('{"processed": true, "delete_this_file": true}')
            except:
                pass
        
        return True
        
    except json.JSONDecodeError as e:
        logger.error("Invalid reset file: not valid JSON", error=str(e))
        return False
    except Exception as e:
        logger.error("Error processing reset file", error=str(e))
        return False


def get_reset_file_instructions() -> dict:
    """Get instructions for creating a reset file."""
    return {
        "instructions": [
            "1. Power off the SeqMaster tester",
            "2. Remove the SD-card and insert it into a computer",
            "3. Create a file named 'seqmaster_reset.json' in the boot partition",
            "4. Add the following content to the file:",
        ],
        "file_content": {
            "action": "reset_admin",
            "new_password": "YOUR_NEW_PASSWORD_HERE",
            "confirm": "I understand this resets the admin password"
        },
        "file_paths": RESET_FILE_PATHS + [str(DATA_RESET_PATH)],
        "notes": [
            "Replace YOUR_NEW_PASSWORD_HERE with your desired password (min 6 characters)",
            "Insert the SD-card back and power on the tester",
            "The file will be automatically deleted after processing",
            "Log in with username 'admin' and your new password"
        ]
    }
