"""
SeqMaster Runtime - Central API Client
Layer: SYNC

HTTP client for communicating with SeqMaster Central server.
"""

import asyncio
from typing import Any, Dict, List, Optional
from datetime import datetime

import httpx
import structlog

from src.core.config import settings

logger = structlog.get_logger(__name__)


class CentralClient:
    """
    HTTP client for SeqMaster Central API.
    
    Handles authentication, retries, and connection management.
    """
    
    def __init__(self, base_url: Optional[str] = None, api_key: Optional[str] = None):
        self.base_url = base_url or settings.CENTRAL_URL
        self.api_key = api_key or settings.CENTRAL_API_KEY
        self._client: Optional[httpx.AsyncClient] = None
    
    @property
    def is_configured(self) -> bool:
        """Check if Central server is configured."""
        return bool(self.base_url)
    
    async def _get_client(self) -> httpx.AsyncClient:
        """Get or create HTTP client."""
        if self._client is None or self._client.is_closed:
            self._client = httpx.AsyncClient(
                base_url=self.base_url,
                timeout=30.0,
                headers={
                    "Content-Type": "application/json",
                    "X-Tester-ID": settings.TESTER_ID,
                    "X-API-Key": self.api_key or "",
                }
            )
        return self._client
    
    async def close(self):
        """Close HTTP client."""
        if self._client and not self._client.is_closed:
            await self._client.aclose()
            self._client = None
    
    async def health_check(self) -> bool:
        """Check if Central server is reachable."""
        if not self.is_configured:
            return False
        
        try:
            client = await self._get_client()
            response = await client.get("/api/v1/health", timeout=5.0)
            return response.status_code == 200
        except Exception as e:
            logger.debug("Central health check failed", error=str(e))
            return False
    
    # ========================================
    # SYNC ENDPOINTS
    # ========================================
    
    async def push_results(self, results: List[Dict[str, Any]]) -> Dict[str, Any]:
        """
        Push test results to Central.
        
        Args:
            results: List of test result payloads
            
        Returns:
            Response with accepted/rejected counts
        """
        if not self.is_configured:
            raise ConnectionError("Central server not configured")
        
        client = await self._get_client()
        response = await client.post(
            "/api/v1/sync/results",
            json={
                "tester_id": settings.TESTER_ID,
                "site_id": settings.SITE_ID,
                "results": results,
                "timestamp": datetime.utcnow().isoformat()
            }
        )
        response.raise_for_status()
        return response.json()
    
    async def push_audit_logs(self, logs: List[Dict[str, Any]]) -> Dict[str, Any]:
        """Push audit logs to Central."""
        if not self.is_configured:
            raise ConnectionError("Central server not configured")
        
        client = await self._get_client()
        response = await client.post(
            "/api/v1/sync/audit-logs",
            json={
                "tester_id": settings.TESTER_ID,
                "logs": logs,
                "timestamp": datetime.utcnow().isoformat()
            }
        )
        response.raise_for_status()
        return response.json()
    
    async def register_tester(self) -> Dict[str, Any]:
        """Register this tester with Central."""
        if not self.is_configured:
            raise ConnectionError("Central server not configured")
        
        from src.core.service_registry import ServiceRegistry
        capabilities = ServiceRegistry.get("capabilities")
        
        client = await self._get_client()
        response = await client.post(
            "/api/v1/testers/register",
            json={
                "tester_id": settings.TESTER_ID,
                "hostname": settings.HOSTNAME,
                "site_id": settings.SITE_ID,
                "version": settings.VERSION,
                "capabilities": capabilities.to_dict() if capabilities else {}
            }
        )
        response.raise_for_status()
        return response.json()
    
    async def heartbeat(self) -> Dict[str, Any]:
        """Send heartbeat to Central."""
        if not self.is_configured:
            raise ConnectionError("Central server not configured")
        
        client = await self._get_client()
        response = await client.post(
            f"/api/v1/testers/{settings.TESTER_ID}/heartbeat",
            json={
                "status": "online",
                "timestamp": datetime.utcnow().isoformat()
            }
        )
        response.raise_for_status()
        return response.json()
    
    # ========================================
    # SEQUENCE ENDPOINTS
    # ========================================
    
    async def get_sequences(self, since: Optional[datetime] = None) -> List[Dict[str, Any]]:
        """
        Get sequences from Central.
        
        Args:
            since: Only get sequences updated after this time
            
        Returns:
            List of sequence metadata
        """
        if not self.is_configured:
            raise ConnectionError("Central server not configured")
        
        client = await self._get_client()
        params = {}
        if since:
            params["since"] = since.isoformat()
        
        response = await client.get("/api/v1/sequences", params=params)
        response.raise_for_status()
        return response.json()
    
    async def get_sequence_content(self, sequence_id: str, version: str) -> Dict[str, Any]:
        """Get full sequence content from Central."""
        if not self.is_configured:
            raise ConnectionError("Central server not configured")
        
        client = await self._get_client()
        response = await client.get(
            f"/api/v1/sequences/{sequence_id}",
            params={"version": version}
        )
        response.raise_for_status()
        return response.json()
    
    # ========================================
    # AUTH ENDPOINTS
    # ========================================
    
    async def validate_token(self, token: str) -> Dict[str, Any]:
        """Validate a JWT token with Central."""
        if not self.is_configured:
            raise ConnectionError("Central server not configured")
        
        client = await self._get_client()
        response = await client.post(
            "/api/v1/auth/validate",
            json={"token": token}
        )
        response.raise_for_status()
        return response.json()
    
    async def get_public_key(self) -> str:
        """Get Central's public key for offline JWT validation."""
        if not self.is_configured:
            raise ConnectionError("Central server not configured")
        
        client = await self._get_client()
        response = await client.get("/api/v1/auth/public-key")
        response.raise_for_status()
        return response.json()["public_key"]
