"""
SeqMaster Runtime - Sync Queue Manager
Layer: SYNC

Manages the queue of pending items to sync with Central.
"""

from datetime import datetime
from typing import Any, Dict, List, Optional

import structlog
from sqlalchemy import select, update, delete
from sqlalchemy.ext.asyncio import AsyncSession

from src.database.models import SyncQueue, SyncStatus
from src.database.connection import get_session

logger = structlog.get_logger(__name__)


class SyncQueueManager:
    """
    Manages the sync queue for offline-first architecture.
    
    Provides methods to:
    - Enqueue items for sync
    - Get pending items
    - Mark items as synced or failed
    - Retry failed items
    """
    
    def __init__(self, session: Optional[AsyncSession] = None):
        self._session = session
    
    async def _get_session(self) -> AsyncSession:
        """Get database session."""
        if self._session:
            return self._session
        async with get_session() as session:
            return session
    
    async def enqueue(
        self,
        entity_type: str,
        entity_id: str,
        payload: Dict[str, Any]
    ) -> SyncQueue:
        """
        Add an item to the sync queue.
        
        Args:
            entity_type: Type of entity ('result', 'audit_log')
            entity_id: Unique ID of the entity
            payload: Data to send to Central
            
        Returns:
            Created SyncQueue item
        """
        async with get_session() as session:
            # Check if already queued
            existing = await session.execute(
                select(SyncQueue).where(
                    SyncQueue.entity_type == entity_type,
                    SyncQueue.entity_id == entity_id,
                    SyncQueue.status.in_([SyncStatus.PENDING, SyncStatus.IN_PROGRESS])
                )
            )
            if existing.scalar_one_or_none():
                logger.debug("Item already in sync queue", 
                           entity_type=entity_type, entity_id=entity_id)
                return existing.scalar_one_or_none()
            
            # Create new queue item
            item = SyncQueue(
                entity_type=entity_type,
                entity_id=entity_id,
                payload=payload,
                status=SyncStatus.PENDING,
                created_at=datetime.utcnow()
            )
            session.add(item)
            await session.commit()
            
            logger.info("Enqueued for sync", 
                       entity_type=entity_type, entity_id=entity_id)
            return item
    
    async def get_pending(
        self,
        entity_type: Optional[str] = None,
        limit: int = 100
    ) -> List[SyncQueue]:
        """
        Get pending items from queue.
        
        Args:
            entity_type: Filter by entity type
            limit: Maximum items to return
            
        Returns:
            List of pending SyncQueue items
        """
        async with get_session() as session:
            query = select(SyncQueue).where(
                SyncQueue.status == SyncStatus.PENDING
            ).order_by(SyncQueue.created_at).limit(limit)
            
            if entity_type:
                query = query.where(SyncQueue.entity_type == entity_type)
            
            result = await session.execute(query)
            return list(result.scalars().all())
    
    async def mark_in_progress(self, item_ids: List[int]) -> None:
        """Mark items as in progress."""
        async with get_session() as session:
            await session.execute(
                update(SyncQueue)
                .where(SyncQueue.id.in_(item_ids))
                .values(
                    status=SyncStatus.IN_PROGRESS,
                    last_attempt_at=datetime.utcnow()
                )
            )
            await session.commit()
    
    async def mark_completed(self, item_ids: List[int]) -> None:
        """Mark items as successfully synced."""
        async with get_session() as session:
            await session.execute(
                update(SyncQueue)
                .where(SyncQueue.id.in_(item_ids))
                .values(
                    status=SyncStatus.COMPLETED,
                    completed_at=datetime.utcnow()
                )
            )
            await session.commit()
            
            logger.info("Marked items as synced", count=len(item_ids))
    
    async def mark_failed(
        self,
        item_ids: List[int],
        error: str
    ) -> None:
        """Mark items as failed with error message."""
        async with get_session() as session:
            # Get current retry counts
            result = await session.execute(
                select(SyncQueue).where(SyncQueue.id.in_(item_ids))
            )
            items = list(result.scalars().all())
            
            for item in items:
                item.retry_count += 1
                item.last_error = error
                item.last_attempt_at = datetime.utcnow()
                
                # If max retries exceeded, mark as permanently failed
                if item.retry_count >= item.max_retries:
                    item.status = SyncStatus.FAILED
                    logger.warning("Sync item exceeded max retries",
                                 entity_type=item.entity_type,
                                 entity_id=item.entity_id)
                else:
                    item.status = SyncStatus.PENDING  # Will be retried
            
            await session.commit()
    
    async def get_queue_stats(self) -> Dict[str, int]:
        """Get queue statistics."""
        async with get_session() as session:
            stats = {}
            for status in SyncStatus:
                result = await session.execute(
                    select(SyncQueue).where(SyncQueue.status == status)
                )
                stats[status.value] = len(list(result.scalars().all()))
            return stats
    
    async def cleanup_completed(self, older_than_hours: int = 24) -> int:
        """
        Remove completed items older than specified hours.
        
        Returns:
            Number of items deleted
        """
        from datetime import timedelta
        
        async with get_session() as session:
            cutoff = datetime.utcnow() - timedelta(hours=older_than_hours)
            
            result = await session.execute(
                delete(SyncQueue).where(
                    SyncQueue.status == SyncStatus.COMPLETED,
                    SyncQueue.completed_at < cutoff
                )
            )
            await session.commit()
            
            deleted = result.rowcount
            if deleted > 0:
                logger.info("Cleaned up completed sync items", count=deleted)
            return deleted
    
    async def retry_failed(self) -> int:
        """
        Reset failed items for retry.
        
        Returns:
            Number of items reset
        """
        async with get_session() as session:
            result = await session.execute(
                update(SyncQueue)
                .where(
                    SyncQueue.status == SyncStatus.FAILED,
                    SyncQueue.retry_count < SyncQueue.max_retries
                )
                .values(
                    status=SyncStatus.PENDING,
                    retry_count=0,
                    last_error=None
                )
            )
            await session.commit()
            
            reset = result.rowcount
            if reset > 0:
                logger.info("Reset failed items for retry", count=reset)
            return reset
