"""
SeqMaster Runtime - Sync Service
Layer: SYNC

Background service for syncing data with SeqMaster Central.
Implements offline-first architecture with push-only sync.
"""

import asyncio
from datetime import datetime
from typing import Dict, Any, Optional, List

import structlog

from src.core.config import settings
from src.sync.client import CentralClient
from src.sync.queue import SyncQueueManager
from src.database.models import SyncStatus

logger = structlog.get_logger(__name__)


class SyncService:
    """
    Background sync service for offline-first architecture.
    
    Responsibilities:
    - Push pending results to Central
    - Pull updated sequences from Central
    - Register tester with Central
    - Send periodic heartbeats
    
    Design principles:
    - Push-only for results (no conflict resolution)
    - Fail gracefully when Central is unavailable
    - Queue items locally for later sync
    """
    
    def __init__(self):
        self.client = CentralClient()
        self.queue = SyncQueueManager()
        self._running = False
        self._task: Optional[asyncio.Task] = None
        self._sync_interval = settings.SYNC_INTERVAL_SECONDS
    
    @property
    def is_enabled(self) -> bool:
        """Check if sync is enabled and configured."""
        return settings.SYNC_ENABLED and self.client.is_configured
    
    async def start(self):
        """Start the background sync service."""
        if not self.is_enabled:
            logger.info("Sync service disabled or not configured")
            return
        
        if self._running:
            logger.warning("Sync service already running")
            return
        
        self._running = True
        self._task = asyncio.create_task(self._sync_loop())
        logger.info("Sync service started", interval=self._sync_interval)
        
        # Initial registration
        await self._register_tester()
    
    async def stop(self):
        """Stop the background sync service."""
        self._running = False
        if self._task:
            self._task.cancel()
            try:
                await self._task
            except asyncio.CancelledError:
                pass
        
        await self.client.close()
        logger.info("Sync service stopped")
    
    async def _sync_loop(self):
        """Main sync loop."""
        while self._running:
            try:
                await self._do_sync()
            except Exception as e:
                logger.error("Sync loop error", error=str(e))
            
            await asyncio.sleep(self._sync_interval)
    
    async def _do_sync(self):
        """Perform one sync cycle."""
        # Check Central availability
        if not await self.client.health_check():
            logger.debug("Central not reachable, skipping sync")
            return
        
        # Sync results
        await self._sync_results()
        
        # Sync audit logs
        await self._sync_audit_logs()
        
        # Send heartbeat
        await self._send_heartbeat()
        
        # Cleanup old completed items
        await self.queue.cleanup_completed()
    
    async def _sync_results(self):
        """Sync pending test results to Central."""
        pending = await self.queue.get_pending(entity_type="result", limit=50)
        if not pending:
            return
        
        logger.info("Syncing results", count=len(pending))
        
        # Mark as in progress
        item_ids = [item.id for item in pending]
        await self.queue.mark_in_progress(item_ids)
        
        try:
            # Batch push to Central
            payloads = [item.payload for item in pending]
            response = await self.client.push_results(payloads)
            
            # Mark as completed
            await self.queue.mark_completed(item_ids)
            logger.info("Results synced successfully",
                       accepted=response.get("accepted", len(pending)))
            
        except Exception as e:
            logger.error("Failed to sync results", error=str(e))
            await self.queue.mark_failed(item_ids, str(e))
    
    async def _sync_audit_logs(self):
        """Sync pending audit logs to Central."""
        pending = await self.queue.get_pending(entity_type="audit_log", limit=100)
        if not pending:
            return
        
        item_ids = [item.id for item in pending]
        await self.queue.mark_in_progress(item_ids)
        
        try:
            payloads = [item.payload for item in pending]
            await self.client.push_audit_logs(payloads)
            await self.queue.mark_completed(item_ids)
            
        except Exception as e:
            logger.error("Failed to sync audit logs", error=str(e))
            await self.queue.mark_failed(item_ids, str(e))
    
    async def _register_tester(self):
        """Register this tester with Central."""
        if not self.is_enabled:
            return
        
        try:
            response = await self.client.register_tester()
            logger.info("Registered with Central", response=response)
        except Exception as e:
            logger.warning("Failed to register with Central", error=str(e))
    
    async def _send_heartbeat(self):
        """Send heartbeat to Central."""
        try:
            await self.client.heartbeat()
        except Exception as e:
            logger.debug("Heartbeat failed", error=str(e))
    
    # ========================================
    # PUBLIC API
    # ========================================
    
    async def enqueue_result(self, session_id: str, result_data: Dict[str, Any]):
        """
        Enqueue a test result for sync.
        
        Called after test execution completes.
        """
        await self.queue.enqueue(
            entity_type="result",
            entity_id=session_id,
            payload={
                "session_id": session_id,
                "tester_id": settings.TESTER_ID,
                "site_id": settings.SITE_ID,
                **result_data
            }
        )
    
    async def enqueue_audit_log(self, log_id: str, log_data: Dict[str, Any]):
        """Enqueue an audit log for sync."""
        await self.queue.enqueue(
            entity_type="audit_log",
            entity_id=log_id,
            payload={
                "log_id": log_id,
                "tester_id": settings.TESTER_ID,
                **log_data
            }
        )
    
    async def force_sync(self) -> Dict[str, Any]:
        """
        Force immediate sync (for manual trigger).
        
        Returns:
            Sync status and counts
        """
        if not self.is_enabled:
            return {"status": "disabled"}
        
        online = await self.client.health_check()
        if not online:
            return {"status": "offline", "message": "Central not reachable"}
        
        await self._do_sync()
        stats = await self.queue.get_queue_stats()
        
        return {
            "status": "completed",
            "queue_stats": stats
        }
    
    async def get_status(self) -> Dict[str, Any]:
        """Get sync service status."""
        stats = await self.queue.get_queue_stats()
        online = await self.client.health_check() if self.is_enabled else False
        
        return {
            "enabled": self.is_enabled,
            "running": self._running,
            "central_url": settings.CENTRAL_URL or "not configured",
            "central_online": online,
            "queue_stats": stats,
            "sync_interval": self._sync_interval
        }
    
    async def pull_sequences(self) -> List[Dict[str, Any]]:
        """
        Pull updated sequences from Central.
        
        Returns:
            List of new/updated sequences
        """
        if not self.is_enabled:
            return []
        
        try:
            # Get last sync time from config
            # For now, just get all sequences
            sequences = await self.client.get_sequences()
            logger.info("Pulled sequences from Central", count=len(sequences))
            return sequences
        except Exception as e:
            logger.error("Failed to pull sequences", error=str(e))
            return []
